<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @package FeedValue
 */

class FeedValue_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @var string
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of the plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Enqueue widget script conditionally.
	 */
	public function enqueue_scripts() {
		// Get active widgets for current page
		$widget_manager = new FeedValue_Widget_Manager();
		$active_widgets = $widget_manager->get_active_widgets();

		if ( empty( $active_widgets ) ) {
			return;
		}

		// Enqueue widget script
		$widget_script_url = apply_filters( 'feedvalue_widget_script_url', FEEDVALUE_WIDGET_SCRIPT_URL );

		wp_enqueue_script(
			'feedvalue-widget',
			$widget_script_url,
			array(),
			$this->version,
			true
		);

		// Add async attribute
		add_filter(
			'script_loader_tag',
			function ( $tag, $handle ) {
				if ( 'feedvalue-widget' === $handle ) {
					return str_replace( ' src', ' async src', $tag );
				}
				return $tag;
			},
			10,
			2
		);
	}

	/**
	 * Render global footer widgets.
	 */
	public function render_global_footer_widgets() {
		$this->render_global_widgets( 'footer' );
	}

	/**
	 * Render global header widgets.
	 */
	public function render_global_header_widgets() {
		$this->render_global_widgets( 'header' );
	}

	/**
	 * Render global widgets for a specific position.
	 *
	 * @param string $position Widget position (header, footer, before_content, after_content).
	 */
	private function render_global_widgets( $position ) {
		$widget_settings = get_option( 'feedvalue_widget_settings', array() );
		$widget_manager  = new FeedValue_Widget_Manager();

		foreach ( $widget_settings as $widget_id => $settings ) {
			// Check if widget has this global position
			$global_position = isset( $settings['global_position'] ) ? $settings['global_position'] : '';

			if ( $global_position !== $position ) {
				continue;
			}

			// Check if widget should be displayed
			if ( ! $widget_manager->should_display( $widget_id ) ) {
				continue;
			}

			// Render widget
			$this->render_widget( $widget_id );
		}
	}

	/**
	 * Render a single widget.
	 *
	 * @param string $widget_id Widget ID.
	 */
	private function render_widget( $widget_id ) {
		do_action( 'feedvalue_before_widget_render', $widget_id );

		$widget_script_url = apply_filters( 'feedvalue_widget_script_url', FEEDVALUE_WIDGET_SCRIPT_URL );

		?>
		<div id="feedvalue-widget-<?php echo esc_attr( $widget_id ); ?>" class="feedvalue-widget-container"></div>
		<script>
			(function() {
				if (typeof window.FeedValue === 'undefined') {
					var script = document.createElement('script');
					script.src = '<?php echo esc_url( $widget_script_url ); ?>';
					script.async = true;
					script.setAttribute('data-widget-id', '<?php echo esc_attr( $widget_id ); ?>');
					document.head.appendChild(script);
				} else {
					window.FeedValue.init('<?php echo esc_attr( $widget_id ); ?>');
				}
			})();
		</script>
		<?php

		do_action( 'feedvalue_after_widget_render', $widget_id );
	}
}
