<?php
/**
 * The core plugin class.
 *
 * @package FeedValue
 */

class FeedValue {

	/**
	 * The loader that's responsible for maintaining and registering all hooks.
	 *
	 * @var FeedValue_Loader
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @var string
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @var string
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 */
	public function __construct() {
		$this->version     = FEEDVALUE_VERSION;
		$this->plugin_name = 'feedvalue';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 */
	private function load_dependencies() {
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-loader.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-api-client.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-cache.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-widget-manager.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-shortcode.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-updater.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-privacy.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-rest-api.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-i18n.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-logger.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'includes/class-feedvalue-admin-notices.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'admin/class-feedvalue-admin.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'admin/class-feedvalue-settings.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'admin/class-feedvalue-ajax.php';
		require_once FEEDVALUE_PLUGIN_DIR . 'public/class-feedvalue-public.php';

		$this->loader = new FeedValue_Loader();

		// Initialize auto-updater.
		$updater = new FeedValue_Updater();
		$updater->init();

		// Initialize privacy/GDPR compliance.
		$privacy = new FeedValue_Privacy();
		$privacy->init();

		// Initialize REST API.
		$rest_api = new FeedValue_REST_API();
		$rest_api->init();

		// Initialize admin notices.
		feedvalue_notices()->init();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 */
	private function set_locale() {
		$i18n = new FeedValue_i18n();
		$i18n->init();
	}

	/**
	 * Register all hooks related to the admin area functionality.
	 */
	private function define_admin_hooks() {
		$plugin_admin    = new FeedValue_Admin( $this->plugin_name, $this->version );
		$plugin_settings = new FeedValue_Settings();
		$plugin_ajax     = new FeedValue_Ajax();

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );
		$this->loader->add_filter( 'plugin_action_links_' . plugin_basename( FEEDVALUE_PLUGIN_FILE ), $plugin_admin, 'add_action_links' );
		$this->loader->add_filter( 'plugin_row_meta', $plugin_admin, 'add_row_meta_links', 10, 2 );

		// Settings
		$this->loader->add_action( 'admin_init', $plugin_settings, 'register_settings' );

		// AJAX handlers
		$this->loader->add_action( 'wp_ajax_feedvalue_test_connection', $plugin_ajax, 'test_connection' );
		$this->loader->add_action( 'wp_ajax_feedvalue_refresh_widgets', $plugin_ajax, 'refresh_widgets' );
		$this->loader->add_action( 'wp_ajax_feedvalue_clear_cache', $plugin_ajax, 'clear_cache' );
		$this->loader->add_action( 'wp_ajax_feedvalue_get_targeting_rules', $plugin_ajax, 'get_targeting_rules' );
		$this->loader->add_action( 'wp_ajax_feedvalue_save_targeting_rules', $plugin_ajax, 'save_targeting_rules' );
		$this->loader->add_action( 'wp_ajax_feedvalue_save_widget_settings', $plugin_ajax, 'save_widget_settings' );

		// Feedback management AJAX handlers
		$this->loader->add_action( 'wp_ajax_feedvalue_get_feedback', $plugin_ajax, 'get_feedback' );
		$this->loader->add_action( 'wp_ajax_feedvalue_get_feedback_single', $plugin_ajax, 'get_feedback_single' );
		$this->loader->add_action( 'wp_ajax_feedvalue_update_feedback_status', $plugin_ajax, 'update_feedback_status' );
		$this->loader->add_action( 'wp_ajax_feedvalue_delete_feedback', $plugin_ajax, 'delete_feedback' );
		$this->loader->add_action( 'wp_ajax_feedvalue_bulk_feedback_action', $plugin_ajax, 'bulk_feedback_action' );

		// Gutenberg block
		$this->loader->add_action( 'init', $this, 'register_gutenberg_block' );
	}

	/**
	 * Register all hooks related to the public-facing functionality.
	 */
	private function define_public_hooks() {
		$plugin_public = new FeedValue_Public( $this->plugin_name, $this->version );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
		$this->loader->add_action( 'wp_footer', $plugin_public, 'render_global_footer_widgets', 100 );
		$this->loader->add_action( 'wp_head', $plugin_public, 'render_global_header_widgets', 100 );

		// Shortcode
		$shortcode = new FeedValue_Shortcode();
		add_shortcode( 'feedvalue', array( $shortcode, 'render_shortcode' ) );
	}

	/**
	 * Register Gutenberg block.
	 */
	public function register_gutenberg_block() {
		if ( ! function_exists( 'register_block_type' ) ) {
			return;
		}

		register_block_type( FEEDVALUE_PLUGIN_DIR . 'blocks/build/feedvalue-block' );
	}

	/**
	 * Run the loader to execute all hooks.
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin.
	 *
	 * @return string
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @return string
	 */
	public function get_version() {
		return $this->version;
	}
}
