<?php
/**
 * Widget targeting and conditional display logic.
 *
 * @package FeedValue
 */

class FeedValue_Widget_Manager {

	/**
	 * Check if widget should be displayed on current page.
	 *
	 * @param string $widget_id The widget ID.
	 * @return bool True if widget should be displayed, false otherwise.
	 */
	public function should_display( $widget_id ) {
		// Check if widget is enabled
		if ( ! $this->is_widget_enabled( $widget_id ) ) {
			return false;
		}

		// Get targeting rules
		$rules = $this->get_targeting_rules( $widget_id );

		if ( empty( $rules ) ) {
			// No rules = show everywhere
			return true;
		}

		$should_display = false;

		// Check inclusion rules (OR logic)
		if ( isset( $rules['inclusion'] ) && ! empty( $rules['inclusion'] ) ) {
			foreach ( $rules['inclusion'] as $rule ) {
				if ( $this->evaluate_rule( $rule ) ) {
					$should_display = true;
					break;
				}
			}
		} else {
			// No inclusion rules = show everywhere
			$should_display = true;
		}

		// Check exclusion rules (override inclusion)
		if ( isset( $rules['exclusion'] ) && ! empty( $rules['exclusion'] ) ) {
			foreach ( $rules['exclusion'] as $rule ) {
				if ( $this->evaluate_rule( $rule ) ) {
					$should_display = false;
					break;
				}
			}
		}

		return apply_filters( 'feedvalue_should_display_widget', $should_display, $widget_id, $rules );
	}

	/**
	 * Check if widget is enabled.
	 *
	 * @param string $widget_id The widget ID.
	 * @return bool True if enabled, false otherwise.
	 */
	private function is_widget_enabled( $widget_id ) {
		$settings = get_option( 'feedvalue_widget_settings', array() );
		return isset( $settings[ $widget_id ]['enabled'] ) && $settings[ $widget_id ]['enabled'];
	}

	/**
	 * Get targeting rules for a widget.
	 *
	 * @param string $widget_id The widget ID.
	 * @return array Targeting rules.
	 */
	public function get_targeting_rules( $widget_id ) {
		$all_rules = get_option( 'feedvalue_targeting_rules', array() );
		return isset( $all_rules[ $widget_id ] ) ? $all_rules[ $widget_id ] : array();
	}

	/**
	 * Save targeting rules for a widget.
	 *
	 * @param string $widget_id The widget ID.
	 * @param array  $rules     Targeting rules.
	 * @return bool True on success, false on failure.
	 */
	public function save_targeting_rules( $widget_id, $rules ) {
		$all_rules               = get_option( 'feedvalue_targeting_rules', array() );
		$all_rules[ $widget_id ] = $rules;

		return update_option( 'feedvalue_targeting_rules', $all_rules );
	}

	/**
	 * Evaluate a single targeting rule.
	 *
	 * @param array $rule Targeting rule.
	 * @return bool True if rule matches, false otherwise.
	 */
	private function evaluate_rule( $rule ) {
		$type = isset( $rule['type'] ) ? $rule['type'] : '';

		switch ( $type ) {
			case 'all_pages':
				return true;

			case 'homepage':
				return is_front_page();

			case 'post_type':
				$post_types = isset( $rule['value'] ) ? (array) $rule['value'] : array();
				return is_singular( $post_types );

			case 'post_ids':
				$post_ids = isset( $rule['value'] ) ? array_map( 'intval', explode( ',', $rule['value'] ) ) : array();
				return is_singular() && in_array( get_the_ID(), $post_ids, true );

			case 'categories':
				$categories = isset( $rule['value'] ) ? (array) $rule['value'] : array();
				return is_category( $categories ) || has_category( $categories );

			case 'tags':
				$tags = isset( $rule['value'] ) ? (array) $rule['value'] : array();
				return is_tag( $tags ) || has_tag( $tags );

			case 'archives':
				return is_archive();

			case 'search':
				return is_search();

			case 'url_pattern':
				$pattern = isset( $rule['value'] ) ? $rule['value'] : '';
				if ( empty( $pattern ) ) {
					return false;
				}
				$current_url = home_url( add_query_arg( array(), $GLOBALS['wp']->request ) );
				return preg_match( '/' . $pattern . '/i', $current_url ) === 1;

			default:
				return false;
		}
	}

	/**
	 * Get all active widgets for the current page.
	 *
	 * Uses locally stored widget settings configured in WordPress admin.
	 * Does NOT call the external API - widget display decisions are made
	 * based on local configuration only.
	 *
	 * @return array Array of widget IDs.
	 */
	public function get_active_widgets() {
		// Use locally stored widget settings (configured in admin)
		$widget_settings = get_option( 'feedvalue_widget_settings', array() );

		if ( empty( $widget_settings ) ) {
			return array();
		}

		$active_widgets = array();

		foreach ( $widget_settings as $widget_id => $settings ) {
			if ( empty( $widget_id ) ) {
				continue;
			}

			// Check if widget should be displayed on this page
			if ( $this->should_display( $widget_id ) ) {
				$active_widgets[] = $widget_id;
			}
		}

		return $active_widgets;
	}
}
