<?php
/**
 * Plugin auto-updater using CDN version manifest.
 *
 * Checks the FeedValue CDN for new versions and integrates with
 * WordPress's built-in plugin update system.
 *
 * @package FeedValue
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * FeedValue Plugin Updater class.
 *
 * Hooks into WordPress update system to check for and apply updates
 * from the FeedValue CDN.
 */
class FeedValue_Updater {

	/**
	 * URL to the version manifest JSON.
	 *
	 * @var string
	 */
	private $version_url = 'https://cdn.feedvalue.com/wordpress/version.json';

	/**
	 * Plugin basename (e.g., "feedvalue/feedvalue.php").
	 *
	 * @var string
	 */
	private $plugin_basename;

	/**
	 * Plugin slug.
	 *
	 * @var string
	 */
	private $plugin_slug = 'feedvalue';

	/**
	 * Cached version info.
	 *
	 * @var object|null
	 */
	private $version_info = null;

	/**
	 * Cache key for transient.
	 *
	 * @var string
	 */
	private $cache_key = 'feedvalue_update_info';

	/**
	 * Cache duration in seconds (6 hours).
	 *
	 * @var int
	 */
	private $cache_duration = 21600;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->plugin_basename = plugin_basename( FEEDVALUE_PLUGIN_FILE );
	}

	/**
	 * Initialize the updater by registering hooks.
	 */
	public function init() {
		// Hook into the plugin update check.
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_for_update' ) );

		// Hook into plugin information popup.
		add_filter( 'plugins_api', array( $this, 'plugin_info' ), 10, 3 );

		// Clean up after update.
		add_action( 'upgrader_process_complete', array( $this, 'clear_cache' ), 10, 2 );

		// Add "Check for updates" link on plugins page.
		add_filter( 'plugin_row_meta', array( $this, 'add_check_update_link' ), 10, 2 );

		// Handle manual update check.
		add_action( 'admin_init', array( $this, 'handle_manual_update_check' ) );
	}

	/**
	 * Check for plugin updates.
	 *
	 * @param object $transient The update_plugins transient.
	 * @return object Modified transient with update info if available.
	 */
	public function check_for_update( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$version_info = $this->get_version_info();

		if ( ! $version_info ) {
			return $transient;
		}

		// Compare versions.
		$current_version = FEEDVALUE_VERSION;
		$remote_version  = $version_info->version;

		if ( version_compare( $remote_version, $current_version, '>' ) ) {
			$transient->response[ $this->plugin_basename ] = (object) array(
				'slug'        => $this->plugin_slug,
				'plugin'      => $this->plugin_basename,
				'new_version' => $remote_version,
				'url'         => 'https://feedvalue.com/wordpress',
				'package'     => $version_info->download_url,
				'icons'       => array(
					'1x' => 'https://cdn.feedvalue.com/wordpress/assets/icon-128x128.png',
					'2x' => 'https://cdn.feedvalue.com/wordpress/assets/icon-256x256.png',
				),
				'banners'     => array(
					'low'  => 'https://cdn.feedvalue.com/wordpress/assets/banner-772x250.png',
					'high' => 'https://cdn.feedvalue.com/wordpress/assets/banner-1544x500.png',
				),
				'tested'      => '6.7',
				'requires'    => '6.0',
				'requires_php' => '8.0',
			);
		} else {
			// No update available, but still add to no_update to show "up to date".
			$transient->no_update[ $this->plugin_basename ] = (object) array(
				'slug'        => $this->plugin_slug,
				'plugin'      => $this->plugin_basename,
				'new_version' => $current_version,
				'url'         => 'https://feedvalue.com/wordpress',
				'package'     => '',
			);
		}

		return $transient;
	}

	/**
	 * Provide plugin information for the "View details" popup.
	 *
	 * @param false|object|array $result The result object or array.
	 * @param string             $action The type of information being requested.
	 * @param object             $args   Plugin API arguments.
	 * @return false|object Plugin info or false if not our plugin.
	 */
	public function plugin_info( $result, $action, $args ) {
		if ( 'plugin_information' !== $action ) {
			return $result;
		}

		if ( $this->plugin_slug !== $args->slug ) {
			return $result;
		}

		$version_info = $this->get_version_info();

		if ( ! $version_info ) {
			return $result;
		}

		return (object) array(
			'name'           => 'FeedValue - User Feedback Widgets',
			'slug'           => $this->plugin_slug,
			'version'        => $version_info->version,
			'author'         => '<a href="https://feedvalue.com">FeedValue</a>',
			'author_profile' => 'https://feedvalue.com',
			'homepage'       => 'https://feedvalue.com/wordpress',
			'download_link'  => $version_info->download_url,
			'sections'       => array(
				'description'  => $this->get_plugin_description(),
				'installation' => $this->get_installation_instructions(),
				'changelog'    => $this->get_changelog( $version_info->version ),
			),
			'banners'        => array(
				'low'  => 'https://cdn.feedvalue.com/wordpress/assets/banner-772x250.png',
				'high' => 'https://cdn.feedvalue.com/wordpress/assets/banner-1544x500.png',
			),
			'icons'          => array(
				'1x' => 'https://cdn.feedvalue.com/wordpress/assets/icon-128x128.png',
				'2x' => 'https://cdn.feedvalue.com/wordpress/assets/icon-256x256.png',
			),
			'tested'         => '6.7',
			'requires'       => '6.0',
			'requires_php'   => '8.0',
			'last_updated'   => $version_info->build_date ?? gmdate( 'Y-m-d H:i:s' ),
		);
	}

	/**
	 * Fetch version info from CDN.
	 *
	 * @param bool $force_refresh Force a fresh fetch, bypassing cache.
	 * @return object|null Version info object or null on failure.
	 */
	private function get_version_info( $force_refresh = false ) {
		// Return cached if available and not forcing refresh.
		if ( ! $force_refresh && null !== $this->version_info ) {
			return $this->version_info;
		}

		// Try to get from transient.
		if ( ! $force_refresh ) {
			$cached = get_transient( $this->cache_key );
			if ( false !== $cached ) {
				$this->version_info = $cached;
				return $this->version_info;
			}
		}

		// Fetch from CDN.
		$response = wp_remote_get(
			$this->version_url,
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json',
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->log_error( 'Failed to fetch version info: ' . $response->get_error_message() );
			return null;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $status_code ) {
			$this->log_error( 'Version check returned status: ' . $status_code );
			return null;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body );

		if ( json_last_error() !== JSON_ERROR_NONE || ! isset( $data->version ) ) {
			$this->log_error( 'Invalid version JSON response' );
			return null;
		}

		// Cache the result.
		set_transient( $this->cache_key, $data, $this->cache_duration );
		$this->version_info = $data;

		return $this->version_info;
	}

	/**
	 * Clear the update cache.
	 *
	 * @param WP_Upgrader $upgrader WP_Upgrader instance.
	 * @param array       $options  Array of update options.
	 */
	public function clear_cache( $upgrader, $options ) {
		if ( 'update' === $options['action'] && 'plugin' === $options['type'] ) {
			delete_transient( $this->cache_key );
			$this->version_info = null;
		}
	}

	/**
	 * Add "Check for updates" link to plugin row.
	 *
	 * @param array  $links Plugin row links.
	 * @param string $file  Plugin file.
	 * @return array Modified links.
	 */
	public function add_check_update_link( $links, $file ) {
		if ( $file !== $this->plugin_basename ) {
			return $links;
		}

		$check_url = wp_nonce_url(
			admin_url( 'plugins.php?feedvalue_check_update=1' ),
			'feedvalue_check_update'
		);

		$links[] = sprintf(
			'<a href="%s">%s</a>',
			esc_url( $check_url ),
			esc_html__( 'Check for updates', 'feedvalue' )
		);

		return $links;
	}

	/**
	 * Handle manual update check request.
	 */
	public function handle_manual_update_check() {
		if ( ! isset( $_GET['feedvalue_check_update'] ) ) {
			return;
		}

		if ( ! current_user_can( 'update_plugins' ) ) {
			return;
		}

		check_admin_referer( 'feedvalue_check_update' );

		// Clear cache and force refresh.
		delete_transient( $this->cache_key );
		delete_site_transient( 'update_plugins' );

		// Force a fresh check.
		$this->version_info = null;
		$version_info = $this->get_version_info( true );

		// Redirect back with a message.
		$redirect_url = admin_url( 'plugins.php' );

		if ( $version_info && version_compare( $version_info->version, FEEDVALUE_VERSION, '>' ) ) {
			$redirect_url = add_query_arg( 'feedvalue_update_available', '1', $redirect_url );
		} else {
			$redirect_url = add_query_arg( 'feedvalue_up_to_date', '1', $redirect_url );
		}

		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * Get plugin description for info popup.
	 *
	 * @return string HTML description.
	 */
	private function get_plugin_description() {
		return '
			<p>FeedValue helps you collect authentic user feedback with customizable widgets that integrate seamlessly with your FeedValue dashboard.</p>
			<h4>Features</h4>
			<ul>
				<li>Easy-to-use feedback widgets</li>
				<li>Gutenberg block for easy placement</li>
				<li>Shortcode support for classic editor</li>
				<li>Customizable targeting rules</li>
				<li>Real-time synchronization with FeedValue dashboard</li>
				<li>Caching for optimal performance</li>
			</ul>
		';
	}

	/**
	 * Get installation instructions for info popup.
	 *
	 * @return string HTML instructions.
	 */
	private function get_installation_instructions() {
		return '
			<ol>
				<li>Upload the plugin files to <code>/wp-content/plugins/feedvalue</code></li>
				<li>Activate the plugin through the "Plugins" screen in WordPress</li>
				<li>Go to FeedValue settings and enter your API key</li>
				<li>Enable the widgets you want to display</li>
				<li>Use the Gutenberg block or shortcode to place widgets</li>
			</ol>
		';
	}

	/**
	 * Get changelog for info popup.
	 *
	 * @param string $version Current version.
	 * @return string HTML changelog.
	 */
	private function get_changelog( $version ) {
		return sprintf(
			'<h4>Version %s</h4><ul><li>Latest release from FeedValue CDN</li><li>See <a href="https://feedvalue.com/changelog" target="_blank">full changelog</a> for details</li></ul>',
			esc_html( $version )
		);
	}

	/**
	 * Log error message.
	 *
	 * @param string $message Error message.
	 */
	private function log_error( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( '[FeedValue Updater] ' . $message );
		}
	}
}
