<?php
/**
 * REST API endpoints for FeedValue plugin.
 *
 * Provides REST API endpoints for the Gutenberg block and other integrations.
 *
 * @package FeedValue
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class FeedValue_REST_API
 *
 * Registers and handles REST API endpoints.
 */
class FeedValue_REST_API {

	/**
	 * REST API namespace.
	 *
	 * @var string
	 */
	const NAMESPACE = 'feedvalue/v1';

	/**
	 * Initialize REST API hooks.
	 */
	public function init() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes.
	 */
	public function register_routes() {
		// Get widgets list endpoint.
		register_rest_route(
			self::NAMESPACE,
			'/widgets',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_widgets' ),
				'permission_callback' => array( $this, 'get_widgets_permissions_check' ),
			)
		);

		// Get single widget endpoint.
		register_rest_route(
			self::NAMESPACE,
			'/widgets/(?P<id>[a-zA-Z0-9-_]+)',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_widget' ),
				'permission_callback' => array( $this, 'get_widgets_permissions_check' ),
				'args'                => array(
					'id' => array(
						'description'       => __( 'Widget ID', 'feedvalue' ),
						'type'              => 'string',
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);

		// Get connection status endpoint.
		register_rest_route(
			self::NAMESPACE,
			'/status',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_status' ),
				'permission_callback' => array( $this, 'get_widgets_permissions_check' ),
			)
		);
	}

	/**
	 * Check if user has permission to access widgets endpoints.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return bool True if user has permission, false otherwise.
	 */
	public function get_widgets_permissions_check( $request ) {
		return current_user_can( 'edit_posts' );
	}

	/**
	 * Get list of widgets from FeedValue API.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object or error.
	 */
	public function get_widgets( $request ) {
		// Check if API key is configured.
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		if ( empty( $encrypted_key ) ) {
			return new WP_Error(
				'feedvalue_no_api_key',
				__( 'FeedValue API key is not configured. Please configure it in the plugin settings.', 'feedvalue' ),
				array( 'status' => 400 )
			);
		}

		// Try to get cached widgets first.
		$cache   = new FeedValue_Cache();
		$widgets = $cache->get( 'widgets_list' );

		if ( false === $widgets ) {
			// Fetch fresh widgets from API.
			$settings   = new FeedValue_Settings();
			$api_key    = $settings->decrypt_api_key( $encrypted_key );
			$api_client = new FeedValue_API_Client();
			$widgets    = $api_client->get_widgets( $api_key );

			if ( is_wp_error( $widgets ) ) {
				return new WP_Error(
					'feedvalue_api_error',
					$widgets->get_error_message(),
					array( 'status' => 500 )
				);
			}

			// Cache the widgets.
			$cache->set( 'widgets_list', $widgets );
		}

		// Format widgets for REST response.
		$formatted_widgets = array();
		foreach ( $widgets as $widget ) {
			$formatted_widgets[] = array(
				'id'          => isset( $widget['id'] ) ? $widget['id'] : '',
				'name'        => isset( $widget['name'] ) ? $widget['name'] : __( 'Unnamed Widget', 'feedvalue' ),
				'type'        => isset( $widget['type'] ) ? $widget['type'] : 'feedback',
				'description' => isset( $widget['description'] ) ? $widget['description'] : '',
				'created_at'  => isset( $widget['created_at'] ) ? $widget['created_at'] : '',
			);
		}

		return new WP_REST_Response(
			array(
				'widgets' => $formatted_widgets,
				'total'   => count( $formatted_widgets ),
			),
			200
		);
	}

	/**
	 * Get single widget details.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object or error.
	 */
	public function get_widget( $request ) {
		$widget_id = $request->get_param( 'id' );

		if ( empty( $widget_id ) ) {
			return new WP_Error(
				'feedvalue_invalid_widget_id',
				__( 'Widget ID is required.', 'feedvalue' ),
				array( 'status' => 400 )
			);
		}

		// Try to get cached widget config.
		$cache  = new FeedValue_Cache();
		$config = $cache->get( 'widget_config_' . $widget_id );

		if ( false === $config ) {
			// Fetch fresh config from API.
			$api_client = new FeedValue_API_Client();
			$config     = $api_client->get_widget_config( $widget_id );

			if ( is_wp_error( $config ) ) {
				return new WP_Error(
					'feedvalue_api_error',
					$config->get_error_message(),
					array( 'status' => 500 )
				);
			}

			// Cache the config.
			$cache->set( 'widget_config_' . $widget_id, $config );
		}

		return new WP_REST_Response( $config, 200 );
	}

	/**
	 * Get connection status.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response Response object.
	 */
	public function get_status( $request ) {
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		$status = array(
			'configured'  => ! empty( $encrypted_key ),
			'cache_ttl'   => get_option( 'feedvalue_cache_ttl', FEEDVALUE_CACHE_TTL ),
			'api_url'     => get_option( 'feedvalue_api_url_override', '' ) ?: FEEDVALUE_API_ENDPOINT,
			'plugin_version' => FEEDVALUE_VERSION,
		);

		if ( ! empty( $encrypted_key ) ) {
			$settings   = new FeedValue_Settings();
			$api_key    = $settings->decrypt_api_key( $encrypted_key );
			$api_client = new FeedValue_API_Client();
			$result     = $api_client->test_connection( $api_key );

			$status['connected'] = ! is_wp_error( $result );
			$status['error']     = is_wp_error( $result ) ? $result->get_error_message() : null;
		} else {
			$status['connected'] = false;
			$status['error']     = __( 'API key not configured', 'feedvalue' );
		}

		return new WP_REST_Response( $status, 200 );
	}
}
