<?php
/**
 * GDPR and privacy compliance functionality.
 *
 * Handles WordPress privacy hooks for GDPR compliance:
 * - Privacy policy suggested text
 * - Personal data export
 * - Personal data erasure
 *
 * @package FeedValue
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class FeedValue_Privacy
 *
 * Implements WordPress privacy hooks for GDPR compliance.
 */
class FeedValue_Privacy {

	/**
	 * Initialize privacy hooks.
	 */
	public function init() {
		// Add privacy policy suggested text.
		add_action( 'admin_init', array( $this, 'add_privacy_policy_content' ) );

		// Register personal data exporter.
		add_filter( 'wp_privacy_personal_data_exporters', array( $this, 'register_exporter' ) );

		// Register personal data eraser.
		add_filter( 'wp_privacy_personal_data_erasers', array( $this, 'register_eraser' ) );
	}

	/**
	 * Add privacy policy suggested text to WordPress Privacy Policy page.
	 */
	public function add_privacy_policy_content() {
		if ( ! function_exists( 'wp_add_privacy_policy_content' ) ) {
			return;
		}

		$content = $this->get_privacy_policy_text();

		wp_add_privacy_policy_content(
			__( 'FeedValue', 'feedvalue' ),
			wp_kses_post( $content )
		);
	}

	/**
	 * Get privacy policy suggested text.
	 *
	 * @return string Privacy policy HTML content.
	 */
	private function get_privacy_policy_text() {
		$policy_text = '<h2>' . esc_html__( 'FeedValue Feedback Widgets', 'feedvalue' ) . '</h2>';

		$policy_text .= '<p>' . esc_html__( 'This site uses FeedValue to collect user feedback through embedded widgets. When you submit feedback through a FeedValue widget, the following data may be collected:', 'feedvalue' ) . '</p>';

		$policy_text .= '<h3>' . esc_html__( 'What Data We Collect', 'feedvalue' ) . '</h3>';
		$policy_text .= '<ul>';
		$policy_text .= '<li>' . esc_html__( 'Feedback text you submit', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'Email address (if you choose to provide it)', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'Page URL where you submitted feedback', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'Browser type and device information (user agent)', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'Date and time of submission', 'feedvalue' ) . '</li>';
		$policy_text .= '</ul>';

		$policy_text .= '<h3>' . esc_html__( 'How We Use Your Data', 'feedvalue' ) . '</h3>';
		$policy_text .= '<p>' . esc_html__( 'Feedback data is transmitted to FeedValue\'s secure servers and is used to help site administrators improve their products and services. Your feedback may be reviewed by site administrators and used to make improvements to this website.', 'feedvalue' ) . '</p>';

		$policy_text .= '<h3>' . esc_html__( 'Data Storage', 'feedvalue' ) . '</h3>';
		$policy_text .= '<p>' . esc_html__( 'Feedback submissions are stored on FeedValue\'s servers, not in the WordPress database. This means:', 'feedvalue' ) . '</p>';
		$policy_text .= '<ul>';
		$policy_text .= '<li>' . esc_html__( 'Your feedback is not stored locally on this website', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'Data requests for feedback must be made through FeedValue', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'FeedValue maintains security and backup procedures for all stored data', 'feedvalue' ) . '</li>';
		$policy_text .= '</ul>';

		$policy_text .= '<h3>' . esc_html__( 'Your Rights', 'feedvalue' ) . '</h3>';
		$policy_text .= '<p>' . esc_html__( 'You have the right to:', 'feedvalue' ) . '</p>';
		$policy_text .= '<ul>';
		$policy_text .= '<li>' . esc_html__( 'Request access to your feedback data', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'Request deletion of your feedback data', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'Opt out of providing feedback at any time', 'feedvalue' ) . '</li>';
		$policy_text .= '</ul>';

		$policy_text .= '<h3>' . esc_html__( 'Third-Party Service', 'feedvalue' ) . '</h3>';
		$policy_text .= '<p>' . sprintf(
			/* translators: %s: FeedValue privacy policy URL */
			esc_html__( 'FeedValue is a third-party service. For more information about how FeedValue handles your data, please review their privacy policy at %s', 'feedvalue' ),
			'<a href="https://feedvalue.com/privacy" target="_blank" rel="noopener noreferrer">https://feedvalue.com/privacy</a>'
		) . '</p>';

		$policy_text .= '<h3>' . esc_html__( 'Contact', 'feedvalue' ) . '</h3>';
		$policy_text .= '<p>' . esc_html__( 'For data-related requests concerning feedback submitted through FeedValue widgets, please contact:', 'feedvalue' ) . '</p>';
		$policy_text .= '<ul>';
		$policy_text .= '<li>' . esc_html__( 'This website\'s administrator for general inquiries', 'feedvalue' ) . '</li>';
		$policy_text .= '<li>' . esc_html__( 'FeedValue support at support@feedvalue.com for data access or deletion requests', 'feedvalue' ) . '</li>';
		$policy_text .= '</ul>';

		return $policy_text;
	}

	/**
	 * Register personal data exporter.
	 *
	 * @param array $exporters Registered exporters.
	 * @return array Modified exporters array.
	 */
	public function register_exporter( $exporters ) {
		$exporters['feedvalue'] = array(
			'exporter_friendly_name' => __( 'FeedValue Plugin Data', 'feedvalue' ),
			'callback'               => array( $this, 'export_personal_data' ),
		);

		return $exporters;
	}

	/**
	 * Export personal data.
	 *
	 * Note: FeedValue does not store user feedback locally in WordPress.
	 * This exporter only exports plugin configuration data, not feedback submissions.
	 * Users must contact FeedValue directly for feedback data export.
	 *
	 * @param string $email_address User email address.
	 * @param int    $page          Page number for pagination.
	 * @return array Export data array.
	 */
	public function export_personal_data( $email_address, $page = 1 ) {
		$export_items = array();

		// FeedValue does not store user feedback in WordPress database.
		// We only export plugin settings that might be associated with a user.

		// Check if this email matches the admin email (only admin settings exist).
		$admin_email = get_option( 'admin_email' );

		if ( $email_address === $admin_email ) {
			// Export admin-configured settings.
			$api_key_exists = ! empty( get_option( 'feedvalue_api_key', '' ) );

			$data = array(
				array(
					'name'  => __( 'Plugin Status', 'feedvalue' ),
					'value' => __( 'Active', 'feedvalue' ),
				),
				array(
					'name'  => __( 'API Key Configured', 'feedvalue' ),
					'value' => $api_key_exists ? __( 'Yes', 'feedvalue' ) : __( 'No', 'feedvalue' ),
				),
			);

			$export_items[] = array(
				'group_id'          => 'feedvalue',
				'group_label'       => __( 'FeedValue Plugin Settings', 'feedvalue' ),
				'group_description' => __( 'Configuration data for the FeedValue feedback widget plugin.', 'feedvalue' ),
				'item_id'           => 'feedvalue-settings',
				'data'              => $data,
			);
		}

		// Add informational note about feedback data.
		$export_items[] = array(
			'group_id'          => 'feedvalue-info',
			'group_label'       => __( 'FeedValue Feedback Data', 'feedvalue' ),
			'group_description' => __( 'Information about feedback submissions.', 'feedvalue' ),
			'item_id'           => 'feedvalue-feedback-info',
			'data'              => array(
				array(
					'name'  => __( 'Feedback Storage', 'feedvalue' ),
					'value' => __( 'Feedback submissions are stored on FeedValue servers, not in WordPress. To request your feedback data, please contact support@feedvalue.com', 'feedvalue' ),
				),
			),
		);

		return array(
			'data' => $export_items,
			'done' => true,
		);
	}

	/**
	 * Register personal data eraser.
	 *
	 * @param array $erasers Registered erasers.
	 * @return array Modified erasers array.
	 */
	public function register_eraser( $erasers ) {
		$erasers['feedvalue'] = array(
			'eraser_friendly_name' => __( 'FeedValue Plugin Data', 'feedvalue' ),
			'callback'             => array( $this, 'erase_personal_data' ),
		);

		return $erasers;
	}

	/**
	 * Erase personal data.
	 *
	 * Note: FeedValue does not store user feedback locally in WordPress.
	 * This eraser clears any cached data but cannot delete feedback from FeedValue servers.
	 * Users must contact FeedValue directly for feedback data deletion.
	 *
	 * @param string $email_address User email address.
	 * @param int    $page          Page number for pagination.
	 * @return array Erasure result array.
	 */
	public function erase_personal_data( $email_address, $page = 1 ) {
		$items_removed  = 0;
		$items_retained = 0;
		$messages       = array();

		// Clear any cached widget data that might contain user-related info.
		$cache = new FeedValue_Cache();
		$cache->clear_all();

		$messages[] = __( 'FeedValue plugin cache has been cleared.', 'feedvalue' );

		// Note about feedback data stored on FeedValue servers.
		$messages[] = __( 'Feedback submissions are stored on FeedValue servers, not in WordPress. To request deletion of your feedback data, please contact support@feedvalue.com', 'feedvalue' );

		// Mark one item as retained (the feedback on FeedValue servers).
		$items_retained = 1;

		return array(
			'items_removed'  => $items_removed,
			'items_retained' => $items_retained,
			'messages'       => $messages,
			'done'           => true,
		);
	}
}
