<?php
/**
 * Logger class for FeedValue plugin.
 *
 * Provides logging functionality that integrates with WordPress debug.log.
 *
 * @package FeedValue
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class FeedValue_Logger
 *
 * Handles logging for the plugin with support for different log levels.
 */
class FeedValue_Logger {

	/**
	 * Log levels.
	 */
	const LEVEL_DEBUG   = 'DEBUG';
	const LEVEL_INFO    = 'INFO';
	const LEVEL_WARNING = 'WARNING';
	const LEVEL_ERROR   = 'ERROR';

	/**
	 * Singleton instance.
	 *
	 * @var FeedValue_Logger
	 */
	private static $instance = null;

	/**
	 * Whether logging is enabled.
	 *
	 * @var bool
	 */
	private $enabled;

	/**
	 * Minimum log level to record.
	 *
	 * @var string
	 */
	private $min_level;

	/**
	 * Log level priorities (higher = more severe).
	 *
	 * @var array
	 */
	private $level_priorities = array(
		self::LEVEL_DEBUG   => 1,
		self::LEVEL_INFO    => 2,
		self::LEVEL_WARNING => 3,
		self::LEVEL_ERROR   => 4,
	);

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->enabled   = defined( 'WP_DEBUG' ) && WP_DEBUG;
		$this->min_level = defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ? self::LEVEL_DEBUG : self::LEVEL_WARNING;

		// Allow filtering of log settings.
		$this->enabled   = apply_filters( 'feedvalue_logging_enabled', $this->enabled );
		$this->min_level = apply_filters( 'feedvalue_min_log_level', $this->min_level );
	}

	/**
	 * Get singleton instance.
	 *
	 * @return FeedValue_Logger Logger instance.
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Log a message.
	 *
	 * @param string $message Log message.
	 * @param string $level   Log level (DEBUG, INFO, WARNING, ERROR).
	 * @param array  $context Additional context data.
	 */
	public function log( $message, $level = self::LEVEL_INFO, $context = array() ) {
		if ( ! $this->enabled ) {
			return;
		}

		// Check if this level should be logged.
		if ( ! $this->should_log( $level ) ) {
			return;
		}

		// Format the log message.
		$formatted = $this->format_message( $message, $level, $context );

		// Write to WordPress debug.log.
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( $formatted );

		// Allow hooking into log events.
		do_action( 'feedvalue_log', $message, $level, $context );
	}

	/**
	 * Log a debug message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context data.
	 */
	public function debug( $message, $context = array() ) {
		$this->log( $message, self::LEVEL_DEBUG, $context );
	}

	/**
	 * Log an info message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context data.
	 */
	public function info( $message, $context = array() ) {
		$this->log( $message, self::LEVEL_INFO, $context );
	}

	/**
	 * Log a warning message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context data.
	 */
	public function warning( $message, $context = array() ) {
		$this->log( $message, self::LEVEL_WARNING, $context );
	}

	/**
	 * Log an error message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context data.
	 */
	public function error( $message, $context = array() ) {
		$this->log( $message, self::LEVEL_ERROR, $context );
	}

	/**
	 * Log an API error.
	 *
	 * @param string   $endpoint API endpoint that failed.
	 * @param WP_Error $error    WP_Error object.
	 * @param array    $context  Additional context data.
	 */
	public function api_error( $endpoint, $error, $context = array() ) {
		$context['endpoint']   = $endpoint;
		$context['error_code'] = $error->get_error_code();

		$this->error( 'API Error: ' . $error->get_error_message(), $context );
	}

	/**
	 * Log a cache event.
	 *
	 * @param string $action Cache action (hit, miss, set, delete).
	 * @param string $key    Cache key.
	 * @param array  $context Additional context data.
	 */
	public function cache( $action, $key, $context = array() ) {
		$context['action'] = $action;
		$context['key']    = $key;

		$this->debug( sprintf( 'Cache %s: %s', $action, $key ), $context );
	}

	/**
	 * Check if a level should be logged.
	 *
	 * @param string $level Log level to check.
	 * @return bool True if level should be logged, false otherwise.
	 */
	private function should_log( $level ) {
		$level_priority     = isset( $this->level_priorities[ $level ] ) ? $this->level_priorities[ $level ] : 0;
		$min_level_priority = isset( $this->level_priorities[ $this->min_level ] ) ? $this->level_priorities[ $this->min_level ] : 0;

		return $level_priority >= $min_level_priority;
	}

	/**
	 * Format log message.
	 *
	 * @param string $message Log message.
	 * @param string $level   Log level.
	 * @param array  $context Additional context data.
	 * @return string Formatted log message.
	 */
	private function format_message( $message, $level, $context ) {
		$timestamp = current_time( 'mysql' );
		$prefix    = sprintf( '[FeedValue][%s]', $level );

		$formatted = sprintf( '%s %s', $prefix, $message );

		// Add context if present.
		if ( ! empty( $context ) ) {
			$context_string = wp_json_encode( $context, JSON_UNESCAPED_SLASHES );
			$formatted     .= ' Context: ' . $context_string;
		}

		return $formatted;
	}

	/**
	 * Enable logging.
	 */
	public function enable() {
		$this->enabled = true;
	}

	/**
	 * Disable logging.
	 */
	public function disable() {
		$this->enabled = false;
	}

	/**
	 * Check if logging is enabled.
	 *
	 * @return bool True if logging is enabled, false otherwise.
	 */
	public function is_enabled() {
		return $this->enabled;
	}

	/**
	 * Set minimum log level.
	 *
	 * @param string $level Minimum log level.
	 */
	public function set_min_level( $level ) {
		if ( isset( $this->level_priorities[ $level ] ) ) {
			$this->min_level = $level;
		}
	}
}

/**
 * Helper function to get logger instance.
 *
 * @return FeedValue_Logger Logger instance.
 */
function feedvalue_logger() {
	return FeedValue_Logger::instance();
}
