<?php
/**
 * Internationalization (i18n) functionality.
 *
 * Handles loading of translation files and localization.
 *
 * @package FeedValue
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class FeedValue_i18n
 *
 * Defines internationalization functionality.
 */
class FeedValue_i18n {

	/**
	 * Text domain for translations.
	 *
	 * @var string
	 */
	const TEXT_DOMAIN = 'feedvalue';

	/**
	 * Initialize i18n hooks.
	 */
	public function init() {
		add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
		add_action( 'init', array( $this, 'set_script_translations' ) );
	}

	/**
	 * Load plugin text domain for translations.
	 *
	 * Looks for translation files in the following locations (in order):
	 * 1. WP_LANG_DIR/plugins/feedvalue-{locale}.mo
	 * 2. plugin_dir/languages/feedvalue-{locale}.mo
	 */
	public function load_textdomain() {
		// Try to load from WordPress languages directory first.
		$loaded = load_plugin_textdomain(
			self::TEXT_DOMAIN,
			false,
			dirname( plugin_basename( FEEDVALUE_PLUGIN_FILE ) ) . '/languages'
		);

		// Log if translations couldn't be loaded (only in debug mode).
		if ( ! $loaded && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			$locale = determine_locale();
			$mofile = WP_LANG_DIR . '/plugins/' . self::TEXT_DOMAIN . '-' . $locale . '.mo';

			if ( ! file_exists( $mofile ) ) {
				$mofile = FEEDVALUE_PLUGIN_DIR . 'languages/' . self::TEXT_DOMAIN . '-' . $locale . '.mo';
			}

			// Only log if we expected translations (non-English locale).
			if ( 'en_US' !== $locale && ! file_exists( $mofile ) ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( sprintf( '[FeedValue] Translation file not found for locale %s', $locale ) );
			}
		}
	}

	/**
	 * Set script translations for Gutenberg block.
	 *
	 * Allows JavaScript files to use wp.i18n functions with proper translations.
	 */
	public function set_script_translations() {
		if ( function_exists( 'wp_set_script_translations' ) ) {
			// Register translations for the Gutenberg block script.
			wp_set_script_translations(
				'feedvalue-block-editor-script',
				self::TEXT_DOMAIN,
				FEEDVALUE_PLUGIN_DIR . 'languages'
			);

			// Register translations for admin scripts.
			wp_set_script_translations(
				'feedvalue-admin',
				self::TEXT_DOMAIN,
				FEEDVALUE_PLUGIN_DIR . 'languages'
			);
		}
	}

	/**
	 * Get the text domain.
	 *
	 * @return string Text domain.
	 */
	public static function get_text_domain() {
		return self::TEXT_DOMAIN;
	}

	/**
	 * Check if translations are loaded.
	 *
	 * @return bool True if translations are loaded, false otherwise.
	 */
	public static function is_loaded() {
		return is_textdomain_loaded( self::TEXT_DOMAIN );
	}

	/**
	 * Get current locale.
	 *
	 * @return string Current locale.
	 */
	public static function get_locale() {
		return determine_locale();
	}

	/**
	 * Get available translations.
	 *
	 * Returns a list of available translation files in the languages directory.
	 *
	 * @return array Array of available locales.
	 */
	public static function get_available_translations() {
		$translations = array();
		$languages_dir = FEEDVALUE_PLUGIN_DIR . 'languages';

		if ( is_dir( $languages_dir ) ) {
			$files = glob( $languages_dir . '/*.mo' );

			if ( $files ) {
				foreach ( $files as $file ) {
					$filename = basename( $file, '.mo' );
					// Extract locale from filename (feedvalue-{locale}.mo).
					if ( strpos( $filename, self::TEXT_DOMAIN . '-' ) === 0 ) {
						$locale = str_replace( self::TEXT_DOMAIN . '-', '', $filename );
						$translations[] = $locale;
					}
				}
			}
		}

		return $translations;
	}
}
