<?php
/**
 * Cache management using WordPress transients.
 *
 * @package FeedValue
 */

class FeedValue_Cache {

	/**
	 * Cache TTL in seconds.
	 *
	 * @var int
	 */
	private $cache_ttl;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->cache_ttl = (int) get_option( 'feedvalue_cache_ttl', FEEDVALUE_CACHE_TTL );
		$this->cache_ttl = apply_filters( 'feedvalue_cache_ttl', $this->cache_ttl );
	}

	/**
	 * Get cached value.
	 *
	 * @param string $key Cache key.
	 * @return mixed|false Cached value or false if not found.
	 */
	public function get( $key ) {
		$cache_key = $this->get_cache_key( $key );
		return get_transient( $cache_key );
	}

	/**
	 * Set cached value.
	 *
	 * @param string $key   Cache key.
	 * @param mixed  $value Value to cache.
	 * @param int    $ttl   Optional. Time to live in seconds. Default is cache_ttl option.
	 * @return bool True on success, false on failure.
	 */
	public function set( $key, $value, $ttl = null ) {
		$cache_key = $this->get_cache_key( $key );
		$ttl       = $ttl ?? $this->cache_ttl;

		return set_transient( $cache_key, $value, $ttl );
	}

	/**
	 * Delete cached value.
	 *
	 * @param string $key Cache key.
	 * @return bool True on success, false on failure.
	 */
	public function delete( $key ) {
		$cache_key = $this->get_cache_key( $key );
		return delete_transient( $cache_key );
	}

	/**
	 * Clear all FeedValue caches.
	 *
	 * @return void
	 */
	public function clear_all() {
		global $wpdb;

		$wpdb->query(
			"DELETE FROM {$wpdb->options}
			WHERE option_name LIKE '_transient_feedvalue_%'
			OR option_name LIKE '_transient_timeout_feedvalue_%'"
		);

		do_action( 'feedvalue_cache_cleared' );
	}

	/**
	 * Generate cache key with prefix.
	 *
	 * @param string $key Cache key.
	 * @return string Prefixed cache key.
	 */
	private function get_cache_key( $key ) {
		return 'feedvalue_' . sanitize_key( $key );
	}
}
