<?php
/**
 * FeedValue API client for communicating with core-api.
 *
 * @package FeedValue
 */

class FeedValue_API_Client {

	/**
	 * API endpoint base URL.
	 *
	 * @var string
	 */
	private $api_endpoint;

	/**
	 * Rate limit retry-after timestamp.
	 *
	 * @var int
	 */
	private static $rate_limit_until = 0;

	/**
	 * Logger instance.
	 *
	 * @var FeedValue_Logger
	 */
	private $logger;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->api_endpoint = apply_filters( 'feedvalue_api_endpoint', FEEDVALUE_API_ENDPOINT );
		$this->logger       = FeedValue_Logger::instance();
	}

	/**
	 * Get User-Agent string for API requests.
	 *
	 * @return string User-Agent header value.
	 */
	private function get_user_agent() {
		global $wp_version;

		return sprintf(
			'FeedValue-WordPress/%s (WordPress/%s; PHP/%s; %s)',
			FEEDVALUE_VERSION,
			$wp_version,
			PHP_VERSION,
			home_url()
		);
	}

	/**
	 * Get common request headers for API calls.
	 *
	 * @param string $api_key Optional API key for authenticated requests.
	 * @return array Headers array.
	 */
	private function get_request_headers( $api_key = '' ) {
		$headers = array(
			'User-Agent'   => $this->get_user_agent(),
			'Accept'       => 'application/json',
			'Content-Type' => 'application/json',
		);

		if ( ! empty( $api_key ) ) {
			$headers['X-API-Key'] = sanitize_text_field( $api_key );
		}

		return apply_filters( 'feedvalue_api_request_headers', $headers, $api_key );
	}

	/**
	 * Check if we're currently rate limited.
	 *
	 * @return bool|int False if not rate limited, or seconds remaining until retry.
	 */
	public function is_rate_limited() {
		if ( self::$rate_limit_until > time() ) {
			return self::$rate_limit_until - time();
		}
		return false;
	}

	/**
	 * Test API key validity.
	 *
	 * @param string $api_key The API key to test.
	 * @return array|WP_Error Response array or WP_Error on failure.
	 */
	public function test_connection( $api_key ) {
		if ( empty( $api_key ) ) {
			$this->logger->warning( 'API key validation failed: empty key provided' );
			return new WP_Error( 'invalid_api_key', __( 'API key is required.', 'feedvalue' ) );
		}

		// Check if we're currently rate limited.
		$rate_limit_error = $this->check_rate_limit_before_request();
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$endpoint = $this->api_endpoint . '/api/v1/widgets';

		$this->logger->debug( 'Testing API connection', array( 'endpoint' => $endpoint ) );

		$response = wp_remote_get(
			$endpoint,
			array(
				'headers' => $this->get_request_headers( $api_key ),
				'timeout' => 10,
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->api_error( $endpoint, $response );
			return $response;
		}

		// Check for rate limiting.
		$rate_limit_error = $this->handle_rate_limit( $response );
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );

		if ( 200 !== $status_code ) {
			if ( 401 === $status_code ) {
				$this->logger->warning( 'API connection test failed: unauthorized', array( 'status' => $status_code ) );
				return new WP_Error( 'unauthorized', __( 'Invalid API key. Please check your FeedValue dashboard.', 'feedvalue' ) );
			}

			$this->logger->error( 'API connection test failed', array( 'status' => $status_code ) );
			return new WP_Error( 'api_error', sprintf( __( 'API error: %d', 'feedvalue' ), $status_code ) );
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Invalid JSON in API response', array( 'json_error' => json_last_error_msg() ) );
			return new WP_Error( 'invalid_json', __( 'Invalid API response.', 'feedvalue' ) );
		}

		$this->logger->info( 'API connection test successful' );

		return array(
			'success' => true,
			'message' => __( 'Connection successful!', 'feedvalue' ),
		);
	}

	/**
	 * Fetch widget list from FeedValue API.
	 *
	 * @param string $api_key The API key.
	 * @return array|WP_Error Array of widgets or WP_Error on failure.
	 */
	public function get_widgets( $api_key ) {
		if ( empty( $api_key ) ) {
			$this->logger->warning( 'Cannot fetch widgets: API key is empty' );
			return new WP_Error( 'invalid_api_key', __( 'API key is required.', 'feedvalue' ) );
		}

		// Check if we're currently rate limited.
		$rate_limit_error = $this->check_rate_limit_before_request();
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$endpoint = $this->api_endpoint . '/api/v1/widgets';

		$this->logger->debug( 'Fetching widgets from API', array( 'endpoint' => $endpoint ) );

		$response = wp_remote_get(
			$endpoint,
			array(
				'headers' => $this->get_request_headers( $api_key ),
				'timeout' => 10,
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->api_error( $endpoint, $response );
			return $response;
		}

		// Check for rate limiting.
		$rate_limit_error = $this->handle_rate_limit( $response );
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );

		if ( 200 !== $status_code ) {
			$this->logger->error( 'Widget fetch failed', array( 'status' => $status_code ) );

			if ( 401 === $status_code ) {
				return new WP_Error( 'unauthorized', __( 'Invalid API key.', 'feedvalue' ) );
			}

			if ( 404 === $status_code ) {
				$this->logger->info( 'No widgets found for this API key' );
				return new WP_Error( 'not_found', __( 'No widgets found.', 'feedvalue' ) );
			}

			return new WP_Error( 'api_error', sprintf( __( 'API error: %d', 'feedvalue' ), $status_code ) );
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Invalid JSON in widgets response', array( 'json_error' => json_last_error_msg() ) );
			return new WP_Error( 'invalid_json', __( 'Invalid API response.', 'feedvalue' ) );
		}

		if ( ! isset( $data['widgets'] ) || ! is_array( $data['widgets'] ) ) {
			$this->logger->debug( 'No widgets in response' );
			return array(); // No widgets found
		}

		$widget_count = count( $data['widgets'] );
		$this->logger->info( 'Successfully fetched widgets', array( 'count' => $widget_count ) );

		return $data['widgets'];
	}

	/**
	 * Get widget configuration from FeedValue API (public endpoint).
	 *
	 * @param string $widget_id The widget ID.
	 * @return array|WP_Error Widget config or WP_Error on failure.
	 */
	public function get_widget_config( $widget_id ) {
		if ( empty( $widget_id ) ) {
			$this->logger->warning( 'Cannot fetch widget config: Widget ID is empty' );
			return new WP_Error( 'invalid_widget_id', __( 'Widget ID is required.', 'feedvalue' ) );
		}

		// Check if we're currently rate limited.
		$rate_limit_error = $this->check_rate_limit_before_request();
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$endpoint = $this->api_endpoint . '/api/v1/widgets/' . sanitize_text_field( $widget_id ) . '/config';

		$this->logger->debug( 'Fetching widget config', array( 'widget_id' => $widget_id, 'endpoint' => $endpoint ) );

		$response = wp_remote_get(
			$endpoint,
			array(
				'headers' => $this->get_request_headers(), // No API key for public endpoint
				'timeout' => 10,
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->api_error( $endpoint, $response, array( 'widget_id' => $widget_id ) );
			return $response;
		}

		// Check for rate limiting.
		$rate_limit_error = $this->handle_rate_limit( $response );
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );

		if ( 200 !== $status_code ) {
			$this->logger->error( 'Widget config fetch failed', array( 'widget_id' => $widget_id, 'status' => $status_code ) );
			return new WP_Error( 'api_error', sprintf( __( 'Widget config error: %d', 'feedvalue' ), $status_code ) );
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Invalid JSON in widget config', array( 'widget_id' => $widget_id, 'json_error' => json_last_error_msg() ) );
			return new WP_Error( 'invalid_json', __( 'Invalid widget config response.', 'feedvalue' ) );
		}

		$this->logger->debug( 'Successfully fetched widget config', array( 'widget_id' => $widget_id ) );

		return $data;
	}

	/**
	 * Get feedback list for the app associated with the API key.
	 *
	 * @param string $api_key The API key.
	 * @param array  $filters Optional filters (status, limit, offset).
	 * @return array|WP_Error Array of feedback or WP_Error on failure.
	 */
	public function get_feedback( $api_key, $filters = array() ) {
		if ( empty( $api_key ) ) {
			$this->logger->warning( 'Cannot fetch feedback: API key is empty' );
			return new WP_Error( 'invalid_api_key', __( 'API key is required.', 'feedvalue' ) );
		}

		// Check if we're currently rate limited.
		$rate_limit_error = $this->check_rate_limit_before_request();
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		// Build query string from filters
		$query_params = array();
		if ( ! empty( $filters['status'] ) ) {
			$query_params['status'] = sanitize_text_field( $filters['status'] );
		}
		if ( ! empty( $filters['limit'] ) ) {
			$query_params['limit'] = absint( $filters['limit'] );
		}
		if ( ! empty( $filters['offset'] ) ) {
			$query_params['offset'] = absint( $filters['offset'] );
		}

		$endpoint = $this->api_endpoint . '/api/v1/feedback';
		if ( ! empty( $query_params ) ) {
			$endpoint .= '?' . http_build_query( $query_params );
		}

		$this->logger->debug( 'Fetching feedback from API', array( 'endpoint' => $endpoint ) );

		$response = wp_remote_get(
			$endpoint,
			array(
				'headers' => $this->get_request_headers( $api_key ),
				'timeout' => 15,
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->api_error( $endpoint, $response );
			return $response;
		}

		// Check for rate limiting.
		$rate_limit_error = $this->handle_rate_limit( $response );
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );

		if ( 200 !== $status_code ) {
			$this->logger->error( 'Feedback fetch failed', array( 'status' => $status_code ) );

			if ( 401 === $status_code ) {
				return new WP_Error( 'unauthorized', __( 'Invalid API key.', 'feedvalue' ) );
			}

			return new WP_Error( 'api_error', sprintf( __( 'API error: %d', 'feedvalue' ), $status_code ) );
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Invalid JSON in feedback response', array( 'json_error' => json_last_error_msg() ) );
			return new WP_Error( 'invalid_json', __( 'Invalid API response.', 'feedvalue' ) );
		}

		$feedback_count = is_array( $data ) ? count( $data ) : 0;
		$this->logger->info( 'Successfully fetched feedback', array( 'count' => $feedback_count ) );

		return $data;
	}

	/**
	 * Get a single feedback submission by ID.
	 *
	 * @param string $api_key       The API key.
	 * @param string $submission_id The feedback submission ID.
	 * @return array|WP_Error Feedback data or WP_Error on failure.
	 */
	public function get_single_feedback( $api_key, $submission_id ) {
		if ( empty( $api_key ) ) {
			return new WP_Error( 'invalid_api_key', __( 'API key is required.', 'feedvalue' ) );
		}

		if ( empty( $submission_id ) ) {
			return new WP_Error( 'invalid_submission_id', __( 'Submission ID is required.', 'feedvalue' ) );
		}

		$rate_limit_error = $this->check_rate_limit_before_request();
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$endpoint = $this->api_endpoint . '/api/v1/feedback/by-id/' . sanitize_text_field( $submission_id );

		$this->logger->debug( 'Fetching single feedback', array( 'submission_id' => $submission_id ) );

		$response = wp_remote_get(
			$endpoint,
			array(
				'headers' => $this->get_request_headers( $api_key ),
				'timeout' => 10,
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->api_error( $endpoint, $response );
			return $response;
		}

		$rate_limit_error = $this->handle_rate_limit( $response );
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );

		if ( 200 !== $status_code ) {
			if ( 404 === $status_code ) {
				return new WP_Error( 'not_found', __( 'Feedback not found.', 'feedvalue' ) );
			}
			if ( 401 === $status_code ) {
				return new WP_Error( 'unauthorized', __( 'Invalid API key.', 'feedvalue' ) );
			}
			return new WP_Error( 'api_error', sprintf( __( 'API error: %d', 'feedvalue' ), $status_code ) );
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			return new WP_Error( 'invalid_json', __( 'Invalid API response.', 'feedvalue' ) );
		}

		return $data;
	}

	/**
	 * Update feedback status.
	 *
	 * @param string $api_key       The API key.
	 * @param string $submission_id The feedback submission ID.
	 * @param string $status        The new status (new, read, resolved, archived).
	 * @return array|WP_Error Updated feedback data or WP_Error on failure.
	 */
	public function update_feedback_status( $api_key, $submission_id, $status ) {
		if ( empty( $api_key ) ) {
			return new WP_Error( 'invalid_api_key', __( 'API key is required.', 'feedvalue' ) );
		}

		if ( empty( $submission_id ) ) {
			return new WP_Error( 'invalid_submission_id', __( 'Submission ID is required.', 'feedvalue' ) );
		}

		$valid_statuses = array( 'new', 'read', 'resolved', 'archived' );
		if ( ! in_array( $status, $valid_statuses, true ) ) {
			return new WP_Error( 'invalid_status', __( 'Invalid status value.', 'feedvalue' ) );
		}

		$rate_limit_error = $this->check_rate_limit_before_request();
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$endpoint = $this->api_endpoint . '/api/v1/feedback/by-id/' . sanitize_text_field( $submission_id );

		$this->logger->debug( 'Updating feedback status', array( 'submission_id' => $submission_id, 'status' => $status ) );

		$response = wp_remote_request(
			$endpoint,
			array(
				'method'  => 'PATCH',
				'headers' => $this->get_request_headers( $api_key ),
				'body'    => wp_json_encode( array( 'status' => $status ) ),
				'timeout' => 10,
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->api_error( $endpoint, $response );
			return $response;
		}

		$rate_limit_error = $this->handle_rate_limit( $response );
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );

		if ( 200 !== $status_code ) {
			if ( 404 === $status_code ) {
				return new WP_Error( 'not_found', __( 'Feedback not found.', 'feedvalue' ) );
			}
			if ( 401 === $status_code ) {
				return new WP_Error( 'unauthorized', __( 'Invalid API key.', 'feedvalue' ) );
			}
			return new WP_Error( 'api_error', sprintf( __( 'API error: %d', 'feedvalue' ), $status_code ) );
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			return new WP_Error( 'invalid_json', __( 'Invalid API response.', 'feedvalue' ) );
		}

		$this->logger->info( 'Feedback status updated', array( 'submission_id' => $submission_id, 'status' => $status ) );

		return $data;
	}

	/**
	 * Delete a feedback submission.
	 *
	 * @param string $api_key       The API key.
	 * @param string $submission_id The feedback submission ID.
	 * @return bool|WP_Error True on success or WP_Error on failure.
	 */
	public function delete_feedback( $api_key, $submission_id ) {
		if ( empty( $api_key ) ) {
			return new WP_Error( 'invalid_api_key', __( 'API key is required.', 'feedvalue' ) );
		}

		if ( empty( $submission_id ) ) {
			return new WP_Error( 'invalid_submission_id', __( 'Submission ID is required.', 'feedvalue' ) );
		}

		$rate_limit_error = $this->check_rate_limit_before_request();
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$endpoint = $this->api_endpoint . '/api/v1/feedback/by-id/' . sanitize_text_field( $submission_id );

		$this->logger->debug( 'Deleting feedback', array( 'submission_id' => $submission_id ) );

		$response = wp_remote_request(
			$endpoint,
			array(
				'method'  => 'DELETE',
				'headers' => $this->get_request_headers( $api_key ),
				'timeout' => 10,
			)
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->api_error( $endpoint, $response );
			return $response;
		}

		$rate_limit_error = $this->handle_rate_limit( $response );
		if ( null !== $rate_limit_error ) {
			return $rate_limit_error;
		}

		$status_code = wp_remote_retrieve_response_code( $response );

		if ( 204 !== $status_code && 200 !== $status_code ) {
			if ( 404 === $status_code ) {
				return new WP_Error( 'not_found', __( 'Feedback not found.', 'feedvalue' ) );
			}
			if ( 401 === $status_code ) {
				return new WP_Error( 'unauthorized', __( 'Invalid API key.', 'feedvalue' ) );
			}
			return new WP_Error( 'api_error', sprintf( __( 'API error: %d', 'feedvalue' ), $status_code ) );
		}

		$this->logger->info( 'Feedback deleted', array( 'submission_id' => $submission_id ) );

		return true;
	}

	/**
	 * Handle rate limit (429) response.
	 *
	 * @param array $response The wp_remote response.
	 * @return WP_Error|null WP_Error if rate limited, null otherwise.
	 */
	private function handle_rate_limit( $response ) {
		$status_code = wp_remote_retrieve_response_code( $response );

		if ( 429 !== $status_code ) {
			return null;
		}

		// Get Retry-After header (in seconds).
		$retry_after = wp_remote_retrieve_header( $response, 'retry-after' );

		// Default to 60 seconds if header not present.
		$retry_seconds = ! empty( $retry_after ) ? absint( $retry_after ) : 60;

		// Cap at 5 minutes to prevent unreasonable waits.
		$retry_seconds = min( $retry_seconds, 300 );

		// Store the retry time.
		self::$rate_limit_until = time() + $retry_seconds;

		$this->logger->warning( 'Rate limited by API', array( 'retry_seconds' => $retry_seconds ) );

		return new WP_Error(
			'rate_limited',
			sprintf(
				/* translators: %d: number of seconds to wait */
				__( 'Too many requests. Please wait %d seconds before trying again.', 'feedvalue' ),
				$retry_seconds
			),
			array(
				'retry_after' => $retry_seconds,
				'retry_at'    => self::$rate_limit_until,
			)
		);
	}

	/**
	 * Check rate limit before making a request.
	 *
	 * @return WP_Error|null WP_Error if still rate limited, null if OK to proceed.
	 */
	private function check_rate_limit_before_request() {
		$remaining = $this->is_rate_limited();

		if ( false !== $remaining ) {
			return new WP_Error(
				'rate_limited',
				sprintf(
					/* translators: %d: number of seconds to wait */
					__( 'Rate limit in effect. Please wait %d seconds before trying again.', 'feedvalue' ),
					$remaining
				),
				array(
					'retry_after' => $remaining,
					'retry_at'    => self::$rate_limit_until,
				)
			);
		}

		return null;
	}
}
