<?php
/**
 * Admin notices functionality.
 *
 * Handles display of admin notices for success, error, warning, and info messages.
 *
 * @package FeedValue
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class FeedValue_Admin_Notices
 *
 * Manages admin notices with transient-based persistence.
 */
class FeedValue_Admin_Notices {

	/**
	 * Notice types.
	 */
	const TYPE_SUCCESS = 'success';
	const TYPE_ERROR   = 'error';
	const TYPE_WARNING = 'warning';
	const TYPE_INFO    = 'info';

	/**
	 * Transient key for storing notices.
	 *
	 * @var string
	 */
	const TRANSIENT_KEY = 'feedvalue_admin_notices';

	/**
	 * Singleton instance.
	 *
	 * @var FeedValue_Admin_Notices
	 */
	private static $instance = null;

	/**
	 * Constructor.
	 */
	private function __construct() {
		// Private constructor for singleton.
	}

	/**
	 * Get singleton instance.
	 *
	 * @return FeedValue_Admin_Notices Instance.
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Initialize admin notices hooks.
	 */
	public function init() {
		add_action( 'admin_notices', array( $this, 'display_notices' ) );
	}

	/**
	 * Add a notice to be displayed.
	 *
	 * @param string $message     Notice message.
	 * @param string $type        Notice type (success, error, warning, info).
	 * @param bool   $dismissible Whether notice can be dismissed.
	 * @param string $id          Optional unique ID for the notice.
	 */
	public function add( $message, $type = self::TYPE_INFO, $dismissible = true, $id = '' ) {
		$notices = get_transient( self::TRANSIENT_KEY );

		if ( ! is_array( $notices ) ) {
			$notices = array();
		}

		$notice = array(
			'message'     => $message,
			'type'        => $this->validate_type( $type ),
			'dismissible' => $dismissible,
			'id'          => $id,
		);

		// Avoid duplicate notices with same ID.
		if ( ! empty( $id ) ) {
			foreach ( $notices as $key => $existing ) {
				if ( isset( $existing['id'] ) && $existing['id'] === $id ) {
					$notices[ $key ] = $notice;
					set_transient( self::TRANSIENT_KEY, $notices, 60 );
					return;
				}
			}
		}

		$notices[] = $notice;
		set_transient( self::TRANSIENT_KEY, $notices, 60 );
	}

	/**
	 * Add a success notice.
	 *
	 * @param string $message     Notice message.
	 * @param bool   $dismissible Whether notice can be dismissed.
	 * @param string $id          Optional unique ID.
	 */
	public function success( $message, $dismissible = true, $id = '' ) {
		$this->add( $message, self::TYPE_SUCCESS, $dismissible, $id );
	}

	/**
	 * Add an error notice.
	 *
	 * @param string $message     Notice message.
	 * @param bool   $dismissible Whether notice can be dismissed.
	 * @param string $id          Optional unique ID.
	 */
	public function error( $message, $dismissible = true, $id = '' ) {
		$this->add( $message, self::TYPE_ERROR, $dismissible, $id );
	}

	/**
	 * Add a warning notice.
	 *
	 * @param string $message     Notice message.
	 * @param bool   $dismissible Whether notice can be dismissed.
	 * @param string $id          Optional unique ID.
	 */
	public function warning( $message, $dismissible = true, $id = '' ) {
		$this->add( $message, self::TYPE_WARNING, $dismissible, $id );
	}

	/**
	 * Add an info notice.
	 *
	 * @param string $message     Notice message.
	 * @param bool   $dismissible Whether notice can be dismissed.
	 * @param string $id          Optional unique ID.
	 */
	public function info( $message, $dismissible = true, $id = '' ) {
		$this->add( $message, self::TYPE_INFO, $dismissible, $id );
	}

	/**
	 * Display all queued notices.
	 */
	public function display_notices() {
		// Only show on FeedValue admin pages or plugins page.
		$screen = get_current_screen();
		if ( ! $screen ) {
			return;
		}

		$feedvalue_pages = array( 'toplevel_page_feedvalue', 'feedvalue_page_feedvalue-settings', 'feedvalue_page_feedvalue-help', 'plugins' );
		$show_on_page    = in_array( $screen->id, $feedvalue_pages, true ) || strpos( $screen->id, 'feedvalue' ) !== false;

		if ( ! $show_on_page ) {
			return;
		}

		$notices = get_transient( self::TRANSIENT_KEY );

		if ( empty( $notices ) || ! is_array( $notices ) ) {
			return;
		}

		foreach ( $notices as $notice ) {
			$this->render_notice( $notice );
		}

		// Clear notices after displaying.
		delete_transient( self::TRANSIENT_KEY );
	}

	/**
	 * Render a single notice.
	 *
	 * @param array $notice Notice data.
	 */
	private function render_notice( $notice ) {
		$class = 'notice notice-' . esc_attr( $notice['type'] );

		if ( $notice['dismissible'] ) {
			$class .= ' is-dismissible';
		}

		// Add FeedValue-specific class for styling.
		$class .= ' feedvalue-notice';

		printf(
			'<div class="%1$s"><p><strong>%2$s</strong> %3$s</p></div>',
			esc_attr( $class ),
			esc_html__( 'FeedValue:', 'feedvalue' ),
			wp_kses_post( $notice['message'] )
		);
	}

	/**
	 * Validate notice type.
	 *
	 * @param string $type Notice type.
	 * @return string Validated type.
	 */
	private function validate_type( $type ) {
		$valid_types = array( self::TYPE_SUCCESS, self::TYPE_ERROR, self::TYPE_WARNING, self::TYPE_INFO );

		if ( in_array( $type, $valid_types, true ) ) {
			return $type;
		}

		return self::TYPE_INFO;
	}

	/**
	 * Clear all notices.
	 */
	public function clear() {
		delete_transient( self::TRANSIENT_KEY );
	}

	/**
	 * Check if there are pending notices.
	 *
	 * @return bool True if there are pending notices, false otherwise.
	 */
	public function has_notices() {
		$notices = get_transient( self::TRANSIENT_KEY );
		return ! empty( $notices ) && is_array( $notices );
	}
}

/**
 * Helper function to get admin notices instance.
 *
 * @return FeedValue_Admin_Notices Admin notices instance.
 */
function feedvalue_notices() {
	return FeedValue_Admin_Notices::instance();
}
