<?php
/**
 * Fired during plugin activation.
 *
 * @package FeedValue
 */

class FeedValue_Activator {

	/**
	 * Minimum PHP version required.
	 */
	const MIN_PHP_VERSION = '8.0';

	/**
	 * Minimum WordPress version required.
	 */
	const MIN_WP_VERSION = '6.0';

	/**
	 * Activate the plugin.
	 */
	public static function activate() {
		// Check system requirements before activation.
		self::check_requirements();

		// Set default options
		if ( ! get_option( 'feedvalue_cache_ttl' ) ) {
			add_option( 'feedvalue_cache_ttl', FEEDVALUE_CACHE_TTL );
		}

		// Add default widget settings
		if ( ! get_option( 'feedvalue_widget_settings' ) ) {
			add_option( 'feedvalue_widget_settings', array() );
		}

		// Add default targeting rules
		if ( ! get_option( 'feedvalue_targeting_rules' ) ) {
			add_option( 'feedvalue_targeting_rules', array() );
		}

		// Store plugin version for future upgrade routines.
		update_option( 'feedvalue_version', FEEDVALUE_VERSION );

		// Flush rewrite rules (in case we add custom endpoints in future)
		flush_rewrite_rules();
	}

	/**
	 * Check system requirements.
	 *
	 * Verifies PHP version, WordPress version, and required extensions.
	 * Kills activation with a descriptive error if requirements are not met.
	 */
	private static function check_requirements() {
		$errors = array();

		// Check PHP version.
		if ( version_compare( PHP_VERSION, self::MIN_PHP_VERSION, '<' ) ) {
			$errors[] = sprintf(
				/* translators: 1: Current PHP version, 2: Required PHP version */
				__( 'FeedValue requires PHP %2$s or higher. Your server is running PHP %1$s.', 'feedvalue' ),
				PHP_VERSION,
				self::MIN_PHP_VERSION
			);
		}

		// Check WordPress version.
		global $wp_version;
		if ( version_compare( $wp_version, self::MIN_WP_VERSION, '<' ) ) {
			$errors[] = sprintf(
				/* translators: 1: Current WordPress version, 2: Required WordPress version */
				__( 'FeedValue requires WordPress %2$s or higher. You are running WordPress %1$s.', 'feedvalue' ),
				$wp_version,
				self::MIN_WP_VERSION
			);
		}

		// Check for required PHP extensions.
		if ( ! extension_loaded( 'openssl' ) ) {
			$errors[] = __( 'FeedValue requires the OpenSSL PHP extension for secure API key storage. Please contact your hosting provider.', 'feedvalue' );
		}

		if ( ! extension_loaded( 'json' ) ) {
			$errors[] = __( 'FeedValue requires the JSON PHP extension. Please contact your hosting provider.', 'feedvalue' );
		}

		// If there are errors, display them and abort activation.
		if ( ! empty( $errors ) ) {
			$message = '<h1>' . esc_html__( 'FeedValue Activation Error', 'feedvalue' ) . '</h1>';
			$message .= '<p>' . esc_html__( 'The FeedValue plugin cannot be activated due to the following issues:', 'feedvalue' ) . '</p>';
			$message .= '<ul style="list-style: disc; margin-left: 20px;">';

			foreach ( $errors as $error ) {
				$message .= '<li>' . esc_html( $error ) . '</li>';
			}

			$message .= '</ul>';
			$message .= '<p><a href="' . esc_url( admin_url( 'plugins.php' ) ) . '">' . esc_html__( 'Return to Plugins', 'feedvalue' ) . '</a></p>';

			wp_die(
				wp_kses_post( $message ),
				esc_html__( 'Plugin Activation Error', 'feedvalue' ),
				array(
					'back_link' => true,
					'response'  => 200,
				)
			);
		}
	}

	/**
	 * Check if the current environment meets the minimum requirements.
	 *
	 * Can be used to check requirements without stopping execution.
	 *
	 * @return bool True if all requirements are met, false otherwise.
	 */
	public static function meets_requirements() {
		global $wp_version;

		if ( version_compare( PHP_VERSION, self::MIN_PHP_VERSION, '<' ) ) {
			return false;
		}

		if ( version_compare( $wp_version, self::MIN_WP_VERSION, '<' ) ) {
			return false;
		}

		if ( ! extension_loaded( 'openssl' ) || ! extension_loaded( 'json' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Get requirement check failures.
	 *
	 * @return array Array of failed requirement messages.
	 */
	public static function get_requirement_errors() {
		global $wp_version;
		$errors = array();

		if ( version_compare( PHP_VERSION, self::MIN_PHP_VERSION, '<' ) ) {
			$errors['php'] = sprintf(
				/* translators: 1: Current PHP version, 2: Required PHP version */
				__( 'PHP %2$s+ required (you have %1$s)', 'feedvalue' ),
				PHP_VERSION,
				self::MIN_PHP_VERSION
			);
		}

		if ( version_compare( $wp_version, self::MIN_WP_VERSION, '<' ) ) {
			$errors['wordpress'] = sprintf(
				/* translators: 1: Current WordPress version, 2: Required WordPress version */
				__( 'WordPress %2$s+ required (you have %1$s)', 'feedvalue' ),
				$wp_version,
				self::MIN_WP_VERSION
			);
		}

		if ( ! extension_loaded( 'openssl' ) ) {
			$errors['openssl'] = __( 'OpenSSL extension required', 'feedvalue' );
		}

		if ( ! extension_loaded( 'json' ) ) {
			$errors['json'] = __( 'JSON extension required', 'feedvalue' );
		}

		return $errors;
	}
}
