<?php
/**
 * Widget list table template.
 *
 * @package FeedValue
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

// Get API key
$settings      = new FeedValue_Settings();
$encrypted_key = get_option( 'feedvalue_api_key', '' );

if ( empty( $encrypted_key ) ) {
	?>
	<div class="feedvalue-widget-list-section">
		<div class="feedvalue-empty-state">
			<div class="feedvalue-empty-icon">
				<span class="dashicons dashicons-admin-generic"></span>
			</div>
			<h3><?php esc_html_e( 'Connect Your Account', 'feedvalue' ); ?></h3>
			<p><?php esc_html_e( 'Enter your FeedValue API key above to view and manage your widgets.', 'feedvalue' ); ?></p>
		</div>
	</div>
	<?php
	return;
}

$api_key = $settings->decrypt_api_key( $encrypted_key );

// Get widgets from cache or API
$cache   = new FeedValue_Cache();
$widgets = $cache->get( 'widgets_cache' );

if ( false === $widgets ) {
	$api_client = new FeedValue_API_Client();
	$widgets    = $api_client->get_widgets( $api_key );

	if ( is_wp_error( $widgets ) ) {
		?>
		<div class="feedvalue-widget-list-section">
			<div class="feedvalue-empty-state">
				<div class="feedvalue-empty-icon" style="background: linear-gradient(135deg, rgba(239, 68, 68, 0.15) 0%, rgba(239, 68, 68, 0.05) 100%); border-color: rgba(239, 68, 68, 0.25);">
					<span class="dashicons dashicons-warning" style="color: #ef4444;"></span>
				</div>
				<h3><?php esc_html_e( 'Connection Error', 'feedvalue' ); ?></h3>
				<p><?php echo esc_html( $widgets->get_error_message() ); ?></p>
			</div>
		</div>
		<?php
		return;
	}

	$cache->set( 'widgets_cache', $widgets );
}

if ( empty( $widgets ) ) {
	?>
	<div class="feedvalue-widget-list-section">
		<div class="feedvalue-empty-state">
			<div class="feedvalue-empty-icon">
				<span class="dashicons dashicons-format-chat"></span>
			</div>
			<h3><?php esc_html_e( 'No Widgets Yet', 'feedvalue' ); ?></h3>
			<p>
				<?php
				printf(
					/* translators: %s: FeedValue dashboard URL */
					esc_html__( 'Create your first widget in your %s to get started.', 'feedvalue' ),
					'<a href="https://feedvalue.com/app" target="_blank" rel="noopener noreferrer">' . esc_html__( 'FeedValue dashboard', 'feedvalue' ) . '</a>'
				);
				?>
			</p>
		</div>
	</div>
	<?php
	return;
}

$widget_settings = get_option( 'feedvalue_widget_settings', array() );

// Count widgets with auto-display enabled for status indicator.
$auto_display_count = 0;
foreach ( $widgets as $widget ) {
	$wid = isset( $widget['id'] ) ? $widget['id'] : '';
	if ( ! empty( $widget_settings[ $wid ]['enabled'] ) && ! empty( $widget_settings[ $wid ]['global_position'] ) ) {
		++$auto_display_count;
	}
}
?>

<div class="feedvalue-widget-list-section">
	<div class="feedvalue-section-header">
		<h2>
			<span class="feedvalue-section-icon">
				<span class="dashicons dashicons-screenoptions"></span>
			</span>
			<?php esc_html_e( 'Your Widgets', 'feedvalue' ); ?>
			<?php if ( $auto_display_count > 0 ) : ?>
				<span class="feedvalue-global-status-badge" title="<?php esc_attr_e( 'Widgets with auto-display enabled (targeting rules may limit where they appear)', 'feedvalue' ); ?>">
					<span class="dashicons dashicons-admin-site-alt3"></span>
					<?php
					printf(
						/* translators: %d: number of widgets with auto-display enabled */
						esc_html( _n( '%d Auto', '%d Auto', $auto_display_count, 'feedvalue' ) ),
						$auto_display_count
					);
					?>
				</span>
			<?php endif; ?>
		</h2>
		<button type="button" id="feedvalue-refresh-widgets" class="button">
			<span class="dashicons dashicons-update"></span>
			<?php esc_html_e( 'Refresh', 'feedvalue' ); ?>
		</button>
	</div>
	<p class="feedvalue-section-description"><?php esc_html_e( 'Manage your FeedValue widgets and configure where they appear on your site.', 'feedvalue' ); ?></p>

	<table class="feedvalue-widget-table">
		<thead>
			<tr>
				<th class="feedvalue-col-enable"><?php esc_html_e( 'Status', 'feedvalue' ); ?></th>
				<th class="feedvalue-col-name"><?php esc_html_e( 'Widget', 'feedvalue' ); ?></th>
				<th class="feedvalue-col-position"><?php esc_html_e( 'Auto-Display', 'feedvalue' ); ?></th>
				<th class="feedvalue-col-shortcode"><?php esc_html_e( 'Shortcode', 'feedvalue' ); ?></th>
				<th class="feedvalue-col-actions"><?php esc_html_e( 'Actions', 'feedvalue' ); ?></th>
			</tr>
		</thead>
		<tbody>
			<?php foreach ( $widgets as $widget ) : ?>
				<?php
				$widget_id       = isset( $widget['id'] ) ? $widget['id'] : '';
				$widget_name     = isset( $widget['name'] ) ? $widget['name'] : __( 'Unnamed Widget', 'feedvalue' );
				$is_enabled      = isset( $widget_settings[ $widget_id ]['enabled'] ) && $widget_settings[ $widget_id ]['enabled'];
				$global_position = isset( $widget_settings[ $widget_id ]['global_position'] ) ? $widget_settings[ $widget_id ]['global_position'] : '';
				$is_global       = $is_enabled && ! empty( $global_position );
				$shortcode       = '[feedvalue widget_id="' . esc_attr( $widget_id ) . '"]';
				?>
				<tr data-widget-id="<?php echo esc_attr( $widget_id ); ?>" class="<?php echo $is_global ? 'feedvalue-widget-global-active' : ''; ?>">
					<td class="feedvalue-col-enable">
						<label class="feedvalue-toggle">
							<input
								type="checkbox"
								class="feedvalue-widget-enabled"
								value="1"
								<?php checked( $is_enabled ); ?>
							/>
							<span class="feedvalue-toggle-slider"></span>
						</label>
					</td>
					<td class="feedvalue-col-name">
						<span class="feedvalue-widget-name"><?php echo esc_html( $widget_name ); ?></span>
						<?php if ( $is_global ) : ?>
							<span class="feedvalue-auto-display-indicator" title="<?php esc_attr_e( 'Auto-display enabled (targeting rules may limit visibility)', 'feedvalue' ); ?>">
								<span class="dashicons dashicons-admin-site-alt3"></span>
							</span>
						<?php endif; ?>
					</td>
					<td class="feedvalue-col-position">
						<select class="feedvalue-position-select">
							<option value="" <?php selected( $global_position, '' ); ?>><?php esc_html_e( 'Manual Only', 'feedvalue' ); ?></option>
							<option value="footer" <?php selected( $global_position, 'footer' ); ?>><?php esc_html_e( 'Footer (Recommended)', 'feedvalue' ); ?></option>
							<option value="header" <?php selected( $global_position, 'header' ); ?>><?php esc_html_e( 'Header', 'feedvalue' ); ?></option>
						</select>
					</td>
					<td class="feedvalue-col-shortcode">
						<div class="feedvalue-shortcode-actions">
							<code
								class="feedvalue-shortcode feedvalue-clickable-copy"
								data-shortcode="<?php echo esc_attr( $shortcode ); ?>"
								title="<?php esc_attr_e( 'Click to copy', 'feedvalue' ); ?>"
							><?php echo esc_html( $shortcode ); ?></code>
							<button
								type="button"
								class="feedvalue-shortcode-generator-btn"
								data-widget-id="<?php echo esc_attr( $widget_id ); ?>"
								data-widget-name="<?php echo esc_attr( $widget_name ); ?>"
								title="<?php esc_attr_e( 'Open shortcode generator', 'feedvalue' ); ?>"
							>
								<span class="dashicons dashicons-shortcode"></span>
							</button>
						</div>
					</td>
					<td class="feedvalue-col-actions">
						<button
							type="button"
							class="button button-small feedvalue-configure-targeting"
							data-widget-id="<?php echo esc_attr( $widget_id ); ?>"
							data-widget-name="<?php echo esc_attr( $widget_name ); ?>"
						>
							<span class="dashicons dashicons-filter"></span>
							<?php esc_html_e( 'Targeting', 'feedvalue' ); ?>
						</button>
					</td>
				</tr>
			<?php endforeach; ?>
		</tbody>
	</table>

	<p class="description">
		<?php esc_html_e( 'Enable widgets and set Auto-Display to "Footer" or "Header" for automatic placement. Use Targeting to control which pages show the widget. For manual placement, use the shortcode or Gutenberg block.', 'feedvalue' ); ?>
	</p>
</div>

<!-- Targeting Rules Modal -->
<div id="feedvalue-targeting-modal" class="feedvalue-modal" style="display:none;">
	<div class="feedvalue-modal-content feedvalue-targeting-modal-content">
		<div class="feedvalue-modal-header">
			<div class="feedvalue-modal-header-content">
				<span class="feedvalue-modal-icon">
					<span class="dashicons dashicons-filter"></span>
				</span>
				<div>
					<h2><?php esc_html_e( 'Display Targeting', 'feedvalue' ); ?></h2>
					<span class="feedvalue-modal-subtitle" id="feedvalue-targeting-widget-name"></span>
				</div>
			</div>
			<button type="button" class="feedvalue-modal-close" aria-label="<?php esc_attr_e( 'Close', 'feedvalue' ); ?>">
				<span class="dashicons dashicons-no-alt"></span>
			</button>
		</div>

		<div class="feedvalue-modal-body feedvalue-targeting-body">
			<div class="feedvalue-targeting-intro">
				<p><?php esc_html_e( 'Control exactly where this widget appears on your site. Add rules below to include or exclude specific pages.', 'feedvalue' ); ?></p>
			</div>

			<!-- Inclusion Rules Section -->
			<div class="feedvalue-targeting-section feedvalue-targeting-include">
				<div class="feedvalue-targeting-section-header">
					<div class="feedvalue-targeting-section-title">
						<span class="feedvalue-targeting-icon feedvalue-targeting-icon-include">
							<span class="dashicons dashicons-visibility"></span>
						</span>
						<div>
							<h3><?php esc_html_e( 'Show Widget On', 'feedvalue' ); ?></h3>
							<p><?php esc_html_e( 'Widget will appear on pages matching any of these rules', 'feedvalue' ); ?></p>
						</div>
					</div>
					<button type="button" class="button feedvalue-add-rule" data-rule-type="inclusion">
						<span class="dashicons dashicons-plus-alt2"></span>
						<?php esc_html_e( 'Add Rule', 'feedvalue' ); ?>
					</button>
				</div>
				<div class="feedvalue-rules-list" id="feedvalue-inclusion-rules">
					<div class="feedvalue-no-rules">
						<span class="dashicons dashicons-info-outline"></span>
						<?php esc_html_e( 'No inclusion rules. Widget will show on all pages (unless excluded below).', 'feedvalue' ); ?>
					</div>
				</div>
			</div>

			<!-- Exclusion Rules Section -->
			<div class="feedvalue-targeting-section feedvalue-targeting-exclude">
				<div class="feedvalue-targeting-section-header">
					<div class="feedvalue-targeting-section-title">
						<span class="feedvalue-targeting-icon feedvalue-targeting-icon-exclude">
							<span class="dashicons dashicons-hidden"></span>
						</span>
						<div>
							<h3><?php esc_html_e( 'Hide Widget On', 'feedvalue' ); ?></h3>
							<p><?php esc_html_e( 'Widget will be hidden on pages matching any of these rules', 'feedvalue' ); ?></p>
						</div>
					</div>
					<button type="button" class="button feedvalue-add-rule" data-rule-type="exclusion">
						<span class="dashicons dashicons-plus-alt2"></span>
						<?php esc_html_e( 'Add Rule', 'feedvalue' ); ?>
					</button>
				</div>
				<div class="feedvalue-rules-list" id="feedvalue-exclusion-rules">
					<div class="feedvalue-no-rules">
						<span class="dashicons dashicons-info-outline"></span>
						<?php esc_html_e( 'No exclusion rules. Widget will not be hidden from any pages.', 'feedvalue' ); ?>
					</div>
				</div>
			</div>

			<!-- Quick Presets -->
			<div class="feedvalue-targeting-presets">
				<h4><?php esc_html_e( 'Quick Presets', 'feedvalue' ); ?></h4>
				<div class="feedvalue-preset-buttons">
					<button type="button" class="feedvalue-preset-btn" data-preset="all-pages">
						<span class="dashicons dashicons-admin-site-alt3"></span>
						<?php esc_html_e( 'All Pages', 'feedvalue' ); ?>
					</button>
					<button type="button" class="feedvalue-preset-btn" data-preset="blog-only">
						<span class="dashicons dashicons-edit"></span>
						<?php esc_html_e( 'Blog Posts Only', 'feedvalue' ); ?>
					</button>
					<button type="button" class="feedvalue-preset-btn" data-preset="homepage-only">
						<span class="dashicons dashicons-admin-home"></span>
						<?php esc_html_e( 'Homepage Only', 'feedvalue' ); ?>
					</button>
					<button type="button" class="feedvalue-preset-btn" data-preset="not-homepage">
						<span class="dashicons dashicons-migrate"></span>
						<?php esc_html_e( 'Exclude Homepage', 'feedvalue' ); ?>
					</button>
				</div>
			</div>
		</div>

		<div class="feedvalue-modal-footer">
			<button type="button" class="button feedvalue-modal-close">
				<?php esc_html_e( 'Cancel', 'feedvalue' ); ?>
			</button>
			<button type="button" class="button button-primary feedvalue-save-targeting">
				<span class="dashicons dashicons-saved"></span>
				<?php esc_html_e( 'Save Targeting Rules', 'feedvalue' ); ?>
			</button>
		</div>
	</div>
</div>

<!-- Rule Template (hidden, used for cloning) -->
<script type="text/template" id="feedvalue-rule-template">
	<div class="feedvalue-rule-card" data-rule-index="{index}">
		<div class="feedvalue-rule-header">
			<select class="feedvalue-rule-type">
				<option value=""><?php esc_html_e( 'Select condition...', 'feedvalue' ); ?></option>
				<option value="all_pages" data-icon="admin-site-alt3"><?php esc_html_e( 'All Pages', 'feedvalue' ); ?></option>
				<option value="homepage" data-icon="admin-home"><?php esc_html_e( 'Homepage', 'feedvalue' ); ?></option>
				<option value="post_type" data-icon="admin-post" data-has-value="true"><?php esc_html_e( 'Post Type', 'feedvalue' ); ?></option>
				<option value="post_ids" data-icon="edit" data-has-value="true"><?php esc_html_e( 'Specific Posts/Pages', 'feedvalue' ); ?></option>
				<option value="categories" data-icon="category" data-has-value="true"><?php esc_html_e( 'Categories', 'feedvalue' ); ?></option>
				<option value="tags" data-icon="tag" data-has-value="true"><?php esc_html_e( 'Tags', 'feedvalue' ); ?></option>
				<option value="archives" data-icon="archive"><?php esc_html_e( 'Archive Pages', 'feedvalue' ); ?></option>
				<option value="search" data-icon="search"><?php esc_html_e( 'Search Results', 'feedvalue' ); ?></option>
				<option value="url_pattern" data-icon="admin-links" data-has-value="true"><?php esc_html_e( 'URL Pattern', 'feedvalue' ); ?></option>
			</select>
			<button type="button" class="feedvalue-remove-rule" title="<?php esc_attr_e( 'Remove rule', 'feedvalue' ); ?>">
				<span class="dashicons dashicons-trash"></span>
			</button>
		</div>
		<div class="feedvalue-rule-value" style="display:none;">
			<!-- Dynamic value input based on rule type -->
		</div>
	</div>
</script>

<!-- Post Type Selector Template -->
<script type="text/template" id="feedvalue-post-type-template">
	<div class="feedvalue-value-field">
		<label><?php esc_html_e( 'Select post types:', 'feedvalue' ); ?></label>
		<div class="feedvalue-checkbox-grid">
			<?php
			$post_types = get_post_types( array( 'public' => true ), 'objects' );
			foreach ( $post_types as $post_type ) :
				if ( 'attachment' === $post_type->name ) {
					continue;
				}
				?>
				<label class="feedvalue-checkbox-item">
					<input type="checkbox" value="<?php echo esc_attr( $post_type->name ); ?>" />
					<span class="feedvalue-checkbox-label">
						<span class="dashicons dashicons-<?php echo esc_attr( 'post' === $post_type->name ? 'admin-post' : ( 'page' === $post_type->name ? 'admin-page' : 'admin-generic' ) ); ?>"></span>
						<?php echo esc_html( $post_type->labels->singular_name ); ?>
					</span>
				</label>
			<?php endforeach; ?>
		</div>
	</div>
</script>

<!-- Post IDs Input Template -->
<script type="text/template" id="feedvalue-post-ids-template">
	<div class="feedvalue-value-field">
		<label><?php esc_html_e( 'Enter post/page IDs (comma-separated):', 'feedvalue' ); ?></label>
		<input type="text" class="feedvalue-text-input" placeholder="<?php esc_attr_e( 'e.g., 1, 42, 156', 'feedvalue' ); ?>" />
		<span class="feedvalue-field-hint"><?php esc_html_e( 'Find IDs in the post/page editor URL', 'feedvalue' ); ?></span>
	</div>
</script>

<!-- Categories Selector Template -->
<script type="text/template" id="feedvalue-categories-template">
	<div class="feedvalue-value-field">
		<label><?php esc_html_e( 'Select categories:', 'feedvalue' ); ?></label>
		<div class="feedvalue-checkbox-grid feedvalue-scrollable">
			<?php
			$categories = get_categories( array( 'hide_empty' => false ) );
			foreach ( $categories as $category ) :
				?>
				<label class="feedvalue-checkbox-item">
					<input type="checkbox" value="<?php echo esc_attr( $category->slug ); ?>" />
					<span class="feedvalue-checkbox-label"><?php echo esc_html( $category->name ); ?></span>
				</label>
			<?php endforeach; ?>
		</div>
	</div>
</script>

<!-- Tags Selector Template -->
<script type="text/template" id="feedvalue-tags-template">
	<div class="feedvalue-value-field">
		<label><?php esc_html_e( 'Select tags:', 'feedvalue' ); ?></label>
		<div class="feedvalue-checkbox-grid feedvalue-scrollable">
			<?php
			$tags = get_tags( array( 'hide_empty' => false ) );
			if ( ! empty( $tags ) ) :
				foreach ( $tags as $tag ) :
					?>
					<label class="feedvalue-checkbox-item">
						<input type="checkbox" value="<?php echo esc_attr( $tag->slug ); ?>" />
						<span class="feedvalue-checkbox-label"><?php echo esc_html( $tag->name ); ?></span>
					</label>
					<?php
				endforeach;
			else :
				?>
				<p class="feedvalue-no-items"><?php esc_html_e( 'No tags found.', 'feedvalue' ); ?></p>
			<?php endif; ?>
		</div>
	</div>
</script>

<!-- URL Pattern Input Template -->
<script type="text/template" id="feedvalue-url-pattern-template">
	<div class="feedvalue-value-field">
		<label><?php esc_html_e( 'Enter URL pattern (regex):', 'feedvalue' ); ?></label>
		<input type="text" class="feedvalue-text-input" placeholder="<?php esc_attr_e( 'e.g., /products/.* or /blog/', 'feedvalue' ); ?>" />
		<span class="feedvalue-field-hint"><?php esc_html_e( 'Use regex patterns to match URL paths', 'feedvalue' ); ?></span>
	</div>
</script>

<!-- Shortcode Generator Modal -->
<div id="feedvalue-shortcode-modal" class="feedvalue-modal" style="display:none;">
	<div class="feedvalue-modal-content feedvalue-shortcode-modal-content">
		<div class="feedvalue-modal-header">
			<div class="feedvalue-modal-header-content">
				<span class="feedvalue-modal-icon">
					<span class="dashicons dashicons-shortcode"></span>
				</span>
				<div>
					<h2><?php esc_html_e( 'Shortcode Generator', 'feedvalue' ); ?></h2>
					<span class="feedvalue-modal-subtitle" id="feedvalue-shortcode-widget-name"></span>
				</div>
			</div>
			<button type="button" class="feedvalue-modal-close" aria-label="<?php esc_attr_e( 'Close', 'feedvalue' ); ?>">
				<span class="dashicons dashicons-no-alt"></span>
			</button>
		</div>

		<div class="feedvalue-modal-body feedvalue-shortcode-body">
			<div class="feedvalue-shortcode-intro">
				<p><?php esc_html_e( 'Customize your shortcode options below, then copy the generated shortcode to use in your posts, pages, or theme templates.', 'feedvalue' ); ?></p>
			</div>

			<!-- Generated Shortcode Preview -->
			<div class="feedvalue-shortcode-preview-section">
				<label><?php esc_html_e( 'Generated Shortcode:', 'feedvalue' ); ?></label>
				<div class="feedvalue-shortcode-preview">
					<code id="feedvalue-generated-shortcode" class="feedvalue-generated-code"></code>
					<button type="button" id="feedvalue-copy-generated" class="button" title="<?php esc_attr_e( 'Copy shortcode', 'feedvalue' ); ?>">
						<span class="dashicons dashicons-clipboard"></span>
						<?php esc_html_e( 'Copy', 'feedvalue' ); ?>
					</button>
				</div>
			</div>

			<!-- PHP Template Code -->
			<div class="feedvalue-shortcode-template-section">
				<label><?php esc_html_e( 'PHP Template Code:', 'feedvalue' ); ?></label>
				<div class="feedvalue-shortcode-preview">
					<code id="feedvalue-generated-php" class="feedvalue-generated-code"></code>
					<button type="button" id="feedvalue-copy-php" class="button" title="<?php esc_attr_e( 'Copy PHP code', 'feedvalue' ); ?>">
						<span class="dashicons dashicons-clipboard"></span>
						<?php esc_html_e( 'Copy', 'feedvalue' ); ?>
					</button>
				</div>
				<span class="feedvalue-field-hint"><?php esc_html_e( 'Use this in your theme template files (e.g., single.php, page.php).', 'feedvalue' ); ?></span>
			</div>

			<!-- Optional Attributes -->
			<div class="feedvalue-shortcode-options">
				<h4><?php esc_html_e( 'Optional Attributes', 'feedvalue' ); ?></h4>

				<div class="feedvalue-option-row">
					<label for="feedvalue-shortcode-class"><?php esc_html_e( 'CSS Class:', 'feedvalue' ); ?></label>
					<input
						type="text"
						id="feedvalue-shortcode-class"
						class="regular-text feedvalue-shortcode-option"
						placeholder="<?php esc_attr_e( 'e.g., my-custom-class', 'feedvalue' ); ?>"
					/>
					<span class="feedvalue-field-hint"><?php esc_html_e( 'Add custom CSS classes to the widget container.', 'feedvalue' ); ?></span>
				</div>

				<div class="feedvalue-option-row">
					<label for="feedvalue-shortcode-id"><?php esc_html_e( 'CSS ID:', 'feedvalue' ); ?></label>
					<input
						type="text"
						id="feedvalue-shortcode-id"
						class="regular-text feedvalue-shortcode-option"
						placeholder="<?php esc_attr_e( 'e.g., my-widget-id', 'feedvalue' ); ?>"
					/>
					<span class="feedvalue-field-hint"><?php esc_html_e( 'Add a custom ID attribute (must be unique on the page).', 'feedvalue' ); ?></span>
				</div>
			</div>

			<!-- Gutenberg Block Info -->
			<div class="feedvalue-shortcode-gutenberg-info">
				<h4>
					<span class="dashicons dashicons-block-default"></span>
					<?php esc_html_e( 'Using Gutenberg?', 'feedvalue' ); ?>
				</h4>
				<p><?php esc_html_e( 'You can also add widgets using the FeedValue block in the Gutenberg editor. Search for "FeedValue" in the block inserter.', 'feedvalue' ); ?></p>
			</div>
		</div>

		<div class="feedvalue-modal-footer">
			<button type="button" class="button feedvalue-modal-close">
				<?php esc_html_e( 'Close', 'feedvalue' ); ?>
			</button>
		</div>
	</div>
</div>
