/**
 * FeedValue Feedback Page JavaScript
 *
 * Handles AJAX-based feedback management including:
 * - Loading feedback list
 * - Filtering and pagination
 * - Status updates
 * - Delete operations
 * - Bulk actions
 *
 * @package FeedValue
 */

(function($) {
	'use strict';

	// Sample data for demonstration when API not configured
	const sampleFeedback = [
		{
			id: 'sample_001',
			created_at: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Homepage Feedback',
			content: 'Absolutely love the new design! The navigation is so intuitive now. Great job on the recent updates.',
			status: 'new',
			submission_metadata: { email: 'sarah.johnson@example.com', rating: 5 }
		},
		{
			id: 'sample_002',
			created_at: new Date(Date.now() - 1.5 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Checkout Survey',
			content: 'Good experience overall. Would be nice to have Apple Pay as a payment option.',
			status: 'read',
			submission_metadata: { email: 'mike.chen@example.com', rating: 4 }
		},
		{
			id: 'sample_003',
			created_at: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Homepage Feedback',
			content: 'The site works fine but loads a bit slow on mobile. Maybe optimize the images?',
			status: 'new',
			submission_metadata: { rating: 3 }
		},
		{
			id: 'sample_004',
			created_at: new Date(Date.now() - 2.5 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Support Widget',
			content: 'Customer support was incredibly helpful! They resolved my issue within minutes.',
			status: 'archived',
			submission_metadata: { email: 'emma.wilson@example.com', rating: 5 }
		},
		{
			id: 'sample_005',
			created_at: new Date(Date.now() - 3 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Product Review',
			content: 'Quality product, fast shipping. The packaging could be more eco-friendly though.',
			status: 'read',
			submission_metadata: { email: 'alex.rodriguez@example.com', rating: 4 }
		},
		{
			id: 'sample_006',
			created_at: new Date(Date.now() - 4 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Checkout Survey',
			content: 'Had trouble applying the discount code. It kept saying invalid even though it was from your email.',
			status: 'new',
			submission_metadata: { email: 'david.thompson@example.com', rating: 2 }
		},
		{
			id: 'sample_007',
			created_at: new Date(Date.now() - 5 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Homepage Feedback',
			content: 'Best online store I\'ve used in a while. Keep up the great work!',
			status: 'read',
			submission_metadata: { rating: 5 }
		},
		{
			id: 'sample_008',
			created_at: new Date(Date.now() - 6 * 24 * 60 * 60 * 1000).toISOString(),
			widget_name: 'Support Widget',
			content: 'Live chat feature is fantastic. Wish it was available 24/7.',
			status: 'archived',
			submission_metadata: { email: 'lisa.park@example.com', rating: 4 }
		}
	];

	// State management
	const state = {
		feedback: [],
		filters: {
			status: '',
			widget: '',
			rating: '',
			search: ''
		},
		pagination: {
			limit: 20,
			offset: 0,
			total: 0
		},
		selectedIds: [],
		loading: false,
		isDemo: false
	};

	// DOM elements cache
	let $table, $tbody, $pagination, $stats, $filters;

	/**
	 * Initialize the feedback page.
	 */
	function init() {
		cacheElements();
		bindEvents();

		// Check if we should show demo data or load real feedback
		if (typeof feedvalueFeedback !== 'undefined' && feedvalueFeedback.isConfigured) {
			loadFeedback();
		} else {
			// Show demo data when API not configured
			loadDemoData();
		}
	}

	/**
	 * Load demo/sample data for demonstration.
	 */
	function loadDemoData() {
		state.isDemo = true;
		state.feedback = sampleFeedback;
		renderFeedback();
		updateStats();
		updatePagination();
		showDemoNotice();
	}

	/**
	 * Show demo data notice.
	 */
	function showDemoNotice() {
		$('.feedvalue-notice-info').show();
	}

	/**
	 * Cache DOM elements for performance.
	 */
	function cacheElements() {
		$table = $('.feedvalue-feedback-table');
		$tbody = $table.find('tbody');
		$pagination = $('.feedvalue-pagination');
		$stats = $('.feedvalue-stats-grid');
		$filters = $('.feedvalue-filters');
	}

	/**
	 * Bind event listeners.
	 */
	function bindEvents() {
		// Filter changes
		$('#feedvalue-filter-status').on('change', handleFilterChange);
		$('#feedvalue-filter-widget').on('change', handleFilterChange);
		$('#feedvalue-filter-rating').on('change', handleFilterChange);
		$('#feedvalue-search').on('input', debounce(handleSearchChange, 300));

		// Table actions
		$tbody.on('click', '.feedvalue-mark-reviewed', handleMarkReviewed);
		$tbody.on('click', '.feedvalue-archive-feedback', handleArchive);
		$tbody.on('click', '.feedvalue-delete-feedback', handleDelete);
		$tbody.on('click', '.feedvalue-expand-feedback', handleExpandFeedback);

		// Checkbox handling
		$('#feedvalue-select-all').on('change', handleSelectAll);
		$tbody.on('change', '.feedvalue-feedback-checkbox', handleCheckboxChange);

		// Bulk actions
		$('#feedvalue-apply-bulk').on('click', handleBulkAction);

		// Pagination
		$pagination.on('click', '.feedvalue-page-btn:not([disabled])', handlePaginationClick);

		// Export
		$('#feedvalue-export-feedback').on('click', handleExport);
	}

	/**
	 * Load feedback from the API.
	 */
	function loadFeedback() {
		if (state.loading) return;

		state.loading = true;
		showLoading();

		const data = {
			action: 'feedvalue_get_feedback',
			nonce: feedvalueFeedback.nonce,
			status: state.filters.status,
			limit: state.pagination.limit,
			offset: state.pagination.offset
		};

		$.post(feedvalueFeedback.ajaxurl, data)
			.done(function(response) {
				if (response.success) {
					state.feedback = response.data.feedback || [];
					renderFeedback();
					updateStats();
					updatePagination();
					hideNotConfiguredNotice();
				} else {
					showError(response.data.message || feedvalueFeedback.strings.error);
					if (response.data.message && response.data.message.includes('API key')) {
						showNotConfiguredNotice();
					}
				}
			})
			.fail(function() {
				showError(feedvalueFeedback.strings.error);
			})
			.always(function() {
				state.loading = false;
				hideLoading();
			});
	}

	/**
	 * Render feedback table rows.
	 */
	function renderFeedback() {
		$tbody.empty();

		if (state.feedback.length === 0) {
			renderEmptyState();
			return;
		}

		// Apply client-side filters (widget, rating, search)
		let filtered = filterFeedback(state.feedback);

		if (filtered.length === 0) {
			renderNoResultsState();
			return;
		}

		filtered.forEach(function(item) {
			$tbody.append(renderFeedbackRow(item));
		});

		updateSelectedCount();
	}

	/**
	 * Filter feedback based on client-side filters.
	 */
	function filterFeedback(feedback) {
		return feedback.filter(function(item) {
			// Widget filter
			if (state.filters.widget && item.widget_name !== state.filters.widget) {
				return false;
			}

			// Rating filter
			if (state.filters.rating) {
				const minRating = parseInt(state.filters.rating, 10);
				const itemRating = item.rating || item.submission_metadata?.rating || 0;
				if (itemRating < minRating) {
					return false;
				}
			}

			// Search filter
			if (state.filters.search) {
				const searchLower = state.filters.search.toLowerCase();
				const content = (item.content || '').toLowerCase();
				const email = (item.submission_metadata?.email || '').toLowerCase();
				if (!content.includes(searchLower) && !email.includes(searchLower)) {
					return false;
				}
			}

			return true;
		});
	}

	/**
	 * Render a single feedback row.
	 */
	function renderFeedbackRow(item) {
		const id = item.id;
		const date = new Date(item.created_at);
		const widgetName = item.widget_name || 'Unknown Widget';
		const rating = item.submission_metadata?.rating || 0;
		const content = item.content || '';
		const email = item.submission_metadata?.email || '';
		const status = item.status || 'new';

		const truncatedContent = truncateWords(content, 15);
		const needsExpand = content.split(' ').length > 15;

		return `
			<tr data-feedback-id="${escapeHtml(id)}" data-status="${escapeHtml(status)}">
				<td class="feedvalue-col-checkbox">
					<input type="checkbox" class="feedvalue-feedback-checkbox" value="${escapeHtml(id)}" />
				</td>
				<td class="feedvalue-col-date">
					<span class="feedvalue-date-primary">${formatDate(date)}</span>
					<span class="feedvalue-date-secondary">${formatTime(date)}</span>
				</td>
				<td class="feedvalue-col-widget">
					<span class="feedvalue-widget-badge">${escapeHtml(widgetName)}</span>
				</td>
				<td class="feedvalue-col-rating">
					${renderStars(rating)}
				</td>
				<td class="feedvalue-col-feedback">
					<div class="feedvalue-feedback-text" data-full-text="${escapeHtml(content)}">
						${escapeHtml(truncatedContent)}
					</div>
					${needsExpand ? '<button type="button" class="feedvalue-expand-feedback">' + feedvalueFeedback.strings.readMore + '</button>' : ''}
				</td>
				<td class="feedvalue-col-user">
					${email ? '<a href="mailto:' + escapeHtml(email) + '" class="feedvalue-user-email">' + escapeHtml(email) + '</a>' : '<span class="feedvalue-anonymous">' + feedvalueFeedback.strings.anonymous + '</span>'}
				</td>
				<td class="feedvalue-col-status">
					${renderStatusBadge(status)}
				</td>
				<td class="feedvalue-col-actions">
					<div class="feedvalue-action-buttons">
						${status === 'new' ? '<button type="button" class="button button-small feedvalue-mark-reviewed" title="' + feedvalueFeedback.strings.markReviewed + '"><span class="dashicons dashicons-yes"></span></button>' : ''}
						${status !== 'archived' ? '<button type="button" class="button button-small feedvalue-archive-feedback" title="' + feedvalueFeedback.strings.archive + '"><span class="dashicons dashicons-archive"></span></button>' : ''}
						<button type="button" class="button button-small feedvalue-delete-feedback" title="${feedvalueFeedback.strings.delete}"><span class="dashicons dashicons-trash"></span></button>
					</div>
				</td>
			</tr>
		`;
	}

	/**
	 * Render empty state when no feedback exists.
	 */
	function renderEmptyState() {
		$tbody.html(`
			<tr class="feedvalue-empty-state">
				<td colspan="8">
					<div class="feedvalue-empty-content">
						<span class="dashicons dashicons-format-chat"></span>
						<h3>${feedvalueFeedback.strings.noFeedback}</h3>
						<p>${feedvalueFeedback.strings.noFeedbackDesc}</p>
					</div>
				</td>
			</tr>
		`);
	}

	/**
	 * Render no results state after filtering.
	 */
	function renderNoResultsState() {
		$tbody.html(`
			<tr class="feedvalue-empty-state">
				<td colspan="8">
					<div class="feedvalue-empty-content">
						<span class="dashicons dashicons-search"></span>
						<h3>${feedvalueFeedback.strings.noResults}</h3>
						<p>${feedvalueFeedback.strings.noResultsDesc}</p>
					</div>
				</td>
			</tr>
		`);
	}

	/**
	 * Update stats cards.
	 */
	function updateStats() {
		const total = state.feedback.length;
		const needsReview = state.feedback.filter(f => f.status === 'new').length;

		// Calculate average rating
		let totalRating = 0;
		let ratingCount = 0;
		state.feedback.forEach(function(f) {
			const rating = f.submission_metadata?.rating;
			if (rating) {
				totalRating += rating;
				ratingCount++;
			}
		});
		const avgRating = ratingCount > 0 ? (totalRating / ratingCount).toFixed(1) : '0.0';

		// This week count
		const oneWeekAgo = new Date();
		oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
		const thisWeek = state.feedback.filter(f => new Date(f.created_at) > oneWeekAgo).length;

		// Update DOM
		$stats.find('.feedvalue-stat-card:nth-child(1) .feedvalue-stat-value').text(total);
		$stats.find('.feedvalue-stat-card:nth-child(2) .feedvalue-stat-value').text(needsReview);
		$stats.find('.feedvalue-stat-card:nth-child(3) .feedvalue-stat-value').text(avgRating);
		$stats.find('.feedvalue-stat-card:nth-child(4) .feedvalue-stat-value').text(thisWeek);

		// Update widget filter options
		updateWidgetFilterOptions();
	}

	/**
	 * Update widget filter dropdown with available widgets.
	 */
	function updateWidgetFilterOptions() {
		const widgets = [...new Set(state.feedback.map(f => f.widget_name).filter(Boolean))];
		const $select = $('#feedvalue-filter-widget');
		const currentValue = $select.val();

		$select.find('option:not(:first)').remove();
		widgets.forEach(function(widget) {
			$select.append(`<option value="${escapeHtml(widget)}">${escapeHtml(widget)}</option>`);
		});

		if (currentValue) {
			$select.val(currentValue);
		}
	}

	/**
	 * Update pagination controls.
	 */
	function updatePagination() {
		const total = filterFeedback(state.feedback).length;
		const start = Math.min(state.pagination.offset + 1, total);
		const end = Math.min(state.pagination.offset + state.pagination.limit, total);

		$pagination.find('.feedvalue-pagination-info').text(
			feedvalueFeedback.strings.showingItems
				.replace('%1$d', start)
				.replace('%2$d', end)
				.replace('%3$d', total)
		);

		// For now, disable pagination controls since we're loading all feedback
		// In a future iteration, implement server-side pagination
		$pagination.find('.feedvalue-page-btn').prop('disabled', true);
	}

	/**
	 * Handle filter dropdown change.
	 */
	function handleFilterChange() {
		state.filters.status = $('#feedvalue-filter-status').val();
		state.filters.widget = $('#feedvalue-filter-widget').val();
		state.filters.rating = $('#feedvalue-filter-rating').val();
		state.pagination.offset = 0;

		// Status filter requires API call; others are client-side
		if (this.id === 'feedvalue-filter-status') {
			loadFeedback();
		} else {
			renderFeedback();
			updatePagination();
		}
	}

	/**
	 * Handle search input change.
	 */
	function handleSearchChange() {
		state.filters.search = $('#feedvalue-search').val();
		renderFeedback();
		updatePagination();
	}

	/**
	 * Handle mark as reviewed action.
	 */
	function handleMarkReviewed() {
		const $row = $(this).closest('tr');
		const id = $row.data('feedback-id');

		if (state.isDemo) {
			// Demo mode - just update UI
			updateDemoStatus(id, 'read', $row);
			return;
		}

		updateFeedbackStatus(id, 'read', $row);
	}

	/**
	 * Handle archive action.
	 */
	function handleArchive() {
		const $row = $(this).closest('tr');
		const id = $row.data('feedback-id');

		if (state.isDemo) {
			// Demo mode - just update UI
			updateDemoStatus(id, 'archived', $row);
			return;
		}

		updateFeedbackStatus(id, 'archived', $row);
	}

	/**
	 * Update demo data status (local only, no API call).
	 */
	function updateDemoStatus(id, status, $row) {
		// Update state
		const feedbackItem = state.feedback.find(f => f.id === id);
		if (feedbackItem) {
			feedbackItem.status = status;
		}

		// Update row
		$row.attr('data-status', status);
		$row.find('.feedvalue-col-status').html(renderStatusBadge(status));

		// Update action buttons
		let actionsHtml = '';
		if (status === 'new') {
			actionsHtml += `<button type="button" class="button button-small feedvalue-mark-reviewed" title="${feedvalueFeedback.strings.markReviewed}"><span class="dashicons dashicons-yes"></span></button>`;
		}
		if (status !== 'archived') {
			actionsHtml += `<button type="button" class="button button-small feedvalue-archive-feedback" title="${feedvalueFeedback.strings.archive}"><span class="dashicons dashicons-archive"></span></button>`;
		}
		actionsHtml += `<button type="button" class="button button-small feedvalue-delete-feedback" title="${feedvalueFeedback.strings.delete}"><span class="dashicons dashicons-trash"></span></button>`;

		$row.find('.feedvalue-action-buttons').html(actionsHtml);

		updateStats();
		showSuccess(feedvalueFeedback.strings.demoUpdated || 'Demo updated (no changes saved)');
	}

	/**
	 * Handle delete action.
	 */
	function handleDelete() {
		const $row = $(this).closest('tr');
		const id = $row.data('feedback-id');

		if (!confirm(feedvalueFeedback.strings.confirmDelete)) {
			return;
		}

		if (state.isDemo) {
			// Demo mode - just remove from UI
			$row.fadeOut(300, function() {
				$(this).remove();
				state.feedback = state.feedback.filter(f => f.id !== id);
				updateStats();
				updatePagination();
				if (state.feedback.length === 0) {
					renderEmptyState();
				}
			});
			showSuccess(feedvalueFeedback.strings.demoDeleted || 'Demo item removed (no changes saved)');
			return;
		}

		const $btn = $(this);
		$btn.prop('disabled', true);

		$.post(feedvalueFeedback.ajaxurl, {
			action: 'feedvalue_delete_feedback',
			nonce: feedvalueFeedback.nonce,
			submission_id: id
		})
		.done(function(response) {
			if (response.success) {
				$row.fadeOut(300, function() {
					$(this).remove();
					// Remove from state
					state.feedback = state.feedback.filter(f => f.id !== id);
					updateStats();
					updatePagination();

					if (state.feedback.length === 0) {
						renderEmptyState();
					}
				});
				showSuccess(response.data.message);
			} else {
				showError(response.data.message);
				$btn.prop('disabled', false);
			}
		})
		.fail(function() {
			showError(feedvalueFeedback.strings.error);
			$btn.prop('disabled', false);
		});
	}

	/**
	 * Update feedback status via API.
	 */
	function updateFeedbackStatus(id, status, $row) {
		const $btns = $row.find('.feedvalue-action-buttons button');
		$btns.prop('disabled', true);

		$.post(feedvalueFeedback.ajaxurl, {
			action: 'feedvalue_update_feedback_status',
			nonce: feedvalueFeedback.nonce,
			submission_id: id,
			status: status
		})
		.done(function(response) {
			if (response.success) {
				// Update state
				const feedbackItem = state.feedback.find(f => f.id === id);
				if (feedbackItem) {
					feedbackItem.status = status;
				}

				// Update row
				$row.attr('data-status', status);
				$row.find('.feedvalue-col-status').html(renderStatusBadge(status));

				// Update action buttons
				let actionsHtml = '';
				if (status === 'new') {
					actionsHtml += `<button type="button" class="button button-small feedvalue-mark-reviewed" title="${feedvalueFeedback.strings.markReviewed}"><span class="dashicons dashicons-yes"></span></button>`;
				}
				if (status !== 'archived') {
					actionsHtml += `<button type="button" class="button button-small feedvalue-archive-feedback" title="${feedvalueFeedback.strings.archive}"><span class="dashicons dashicons-archive"></span></button>`;
				}
				actionsHtml += `<button type="button" class="button button-small feedvalue-delete-feedback" title="${feedvalueFeedback.strings.delete}"><span class="dashicons dashicons-trash"></span></button>`;

				$row.find('.feedvalue-action-buttons').html(actionsHtml);

				updateStats();
				showSuccess(response.data.message);
			} else {
				showError(response.data.message);
			}
		})
		.fail(function() {
			showError(feedvalueFeedback.strings.error);
		})
		.always(function() {
			$btns.prop('disabled', false);
		});
	}

	/**
	 * Handle expand feedback text.
	 */
	function handleExpandFeedback() {
		const $btn = $(this);
		const $text = $btn.prev('.feedvalue-feedback-text');
		const fullText = $text.data('full-text');

		if ($btn.hasClass('expanded')) {
			$text.text(truncateWords(fullText, 15));
			$btn.text(feedvalueFeedback.strings.readMore);
			$btn.removeClass('expanded');
		} else {
			$text.text(fullText);
			$btn.text(feedvalueFeedback.strings.showLess);
			$btn.addClass('expanded');
		}
	}

	/**
	 * Handle select all checkbox.
	 */
	function handleSelectAll() {
		const isChecked = $(this).is(':checked');
		$tbody.find('.feedvalue-feedback-checkbox').prop('checked', isChecked);
		updateSelectedIds();
	}

	/**
	 * Handle individual checkbox change.
	 */
	function handleCheckboxChange() {
		updateSelectedIds();

		// Update select all checkbox state
		const $checkboxes = $tbody.find('.feedvalue-feedback-checkbox');
		const $checked = $checkboxes.filter(':checked');
		$('#feedvalue-select-all').prop('checked', $checkboxes.length > 0 && $checkboxes.length === $checked.length);
	}

	/**
	 * Update selected IDs array.
	 */
	function updateSelectedIds() {
		state.selectedIds = [];
		$tbody.find('.feedvalue-feedback-checkbox:checked').each(function() {
			state.selectedIds.push($(this).val());
		});
		updateSelectedCount();
	}

	/**
	 * Update selected count display.
	 */
	function updateSelectedCount() {
		const count = state.selectedIds.length;
		const $count = $('.feedvalue-selected-count');

		if (count > 0) {
			$count.text(feedvalueFeedback.strings.selectedCount.replace('%d', count)).show();
		} else {
			$count.hide();
		}
	}

	/**
	 * Handle bulk action.
	 */
	function handleBulkAction() {
		const action = $('#feedvalue-bulk-action').val();

		if (!action) {
			showError(feedvalueFeedback.strings.selectAction);
			return;
		}

		if (state.selectedIds.length === 0) {
			showError(feedvalueFeedback.strings.selectItems);
			return;
		}

		if (action === 'delete' && !confirm(feedvalueFeedback.strings.confirmBulkDelete)) {
			return;
		}

		// Demo mode - handle locally
		if (state.isDemo) {
			handleDemoBulkAction(action);
			return;
		}

		const $btn = $('#feedvalue-apply-bulk');
		$btn.prop('disabled', true).text(feedvalueFeedback.strings.processing);

		$.post(feedvalueFeedback.ajaxurl, {
			action: 'feedvalue_bulk_feedback_action',
			nonce: feedvalueFeedback.nonce,
			bulk_action: action,
			submission_ids: state.selectedIds
		})
		.done(function(response) {
			if (response.success) {
				showSuccess(response.data.message);
				// Reload feedback to reflect changes
				loadFeedback();
				// Reset selection
				state.selectedIds = [];
				$('#feedvalue-select-all').prop('checked', false);
				$('#feedvalue-bulk-action').val('');
			} else {
				showError(response.data.message);
			}
		})
		.fail(function() {
			showError(feedvalueFeedback.strings.error);
		})
		.always(function() {
			$btn.prop('disabled', false).text(feedvalueFeedback.strings.apply);
		});
	}

	/**
	 * Handle demo bulk action (local only).
	 */
	function handleDemoBulkAction(action) {
		const statusMap = {
			'read': 'read',
			'resolved': 'resolved',
			'archived': 'archived'
		};

		state.selectedIds.forEach(function(id) {
			if (action === 'delete') {
				state.feedback = state.feedback.filter(f => f.id !== id);
			} else if (statusMap[action]) {
				const item = state.feedback.find(f => f.id === id);
				if (item) {
					item.status = statusMap[action];
				}
			}
		});

		// Re-render
		renderFeedback();
		updateStats();
		updatePagination();

		// Reset selection
		state.selectedIds = [];
		$('#feedvalue-select-all').prop('checked', false);
		$('#feedvalue-bulk-action').val('');

		showSuccess(feedvalueFeedback.strings.demoBulkUpdated || 'Demo items updated (no changes saved)');
	}

	/**
	 * Handle pagination click.
	 */
	function handlePaginationClick() {
		// Placeholder for future server-side pagination
		const $btn = $(this);
		// TODO: Implement when needed
	}

	/**
	 * Handle export to CSV.
	 */
	function handleExport() {
		const filtered = filterFeedback(state.feedback);

		if (filtered.length === 0) {
			showError(feedvalueFeedback.strings.noDataExport);
			return;
		}

		// Build CSV content
		const headers = ['Date', 'Widget', 'Rating', 'Feedback', 'Email', 'Status'];
		const rows = filtered.map(function(item) {
			return [
				new Date(item.created_at).toISOString(),
				item.widget_name || '',
				item.submission_metadata?.rating || '',
				'"' + (item.content || '').replace(/"/g, '""') + '"',
				item.submission_metadata?.email || '',
				item.status || ''
			].join(',');
		});

		const csv = headers.join(',') + '\n' + rows.join('\n');

		// Download
		const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
		const link = document.createElement('a');
		const url = URL.createObjectURL(blob);
		link.setAttribute('href', url);
		link.setAttribute('download', 'feedvalue-feedback-' + new Date().toISOString().split('T')[0] + '.csv');
		link.style.visibility = 'hidden';
		document.body.appendChild(link);
		link.click();
		document.body.removeChild(link);
	}

	// Helper functions

	/**
	 * Show loading state.
	 */
	function showLoading() {
		$tbody.html(`
			<tr class="feedvalue-loading-state">
				<td colspan="8">
					<div class="feedvalue-loading-content">
						<span class="spinner is-active"></span>
						<span>${feedvalueFeedback.strings.loading}</span>
					</div>
				</td>
			</tr>
		`);
	}

	/**
	 * Hide loading state.
	 */
	function hideLoading() {
		$tbody.find('.feedvalue-loading-state').remove();
	}

	/**
	 * Show not configured notice.
	 */
	function showNotConfiguredNotice() {
		$('.feedvalue-notice-info').show();
	}

	/**
	 * Hide not configured notice.
	 */
	function hideNotConfiguredNotice() {
		$('.feedvalue-notice-info').hide();
	}

	/**
	 * Show success message.
	 */
	function showSuccess(message) {
		showNotice(message, 'success');
	}

	/**
	 * Show error message.
	 */
	function showError(message) {
		showNotice(message, 'error');
	}

	/**
	 * Show notice.
	 */
	function showNotice(message, type) {
		// Remove existing notices
		$('.feedvalue-toast').remove();

		const $notice = $(`
			<div class="feedvalue-toast feedvalue-toast-${type}">
				<span class="dashicons dashicons-${type === 'success' ? 'yes' : 'warning'}"></span>
				<span class="feedvalue-toast-message">${escapeHtml(message)}</span>
			</div>
		`);

		$('body').append($notice);

		// Animate in
		setTimeout(function() {
			$notice.addClass('visible');
		}, 10);

		// Auto remove
		setTimeout(function() {
			$notice.removeClass('visible');
			setTimeout(function() {
				$notice.remove();
			}, 300);
		}, 4000);
	}

	/**
	 * Render star rating HTML.
	 */
	function renderStars(rating) {
		let html = `<span class="feedvalue-stars" aria-label="${rating} out of 5 stars">`;
		for (let i = 1; i <= 5; i++) {
			html += `<span class="feedvalue-star ${i <= rating ? 'filled' : ''}">★</span>`;
		}
		html += '</span>';
		return html;
	}

	/**
	 * Render status badge HTML.
	 */
	function renderStatusBadge(status) {
		const classes = {
			'new': 'feedvalue-badge feedvalue-badge-new',
			'read': 'feedvalue-badge feedvalue-badge-reviewed',
			'resolved': 'feedvalue-badge feedvalue-badge-resolved',
			'archived': 'feedvalue-badge feedvalue-badge-archived'
		};
		const labels = {
			'new': feedvalueFeedback.strings.statusNew,
			'read': feedvalueFeedback.strings.statusRead,
			'resolved': feedvalueFeedback.strings.statusResolved,
			'archived': feedvalueFeedback.strings.statusArchived
		};
		return `<span class="${classes[status] || ''}">${labels[status] || status}</span>`;
	}

	/**
	 * Format date.
	 */
	function formatDate(date) {
		return date.toLocaleDateString(undefined, { month: 'short', day: 'numeric', year: 'numeric' });
	}

	/**
	 * Format time.
	 */
	function formatTime(date) {
		return date.toLocaleTimeString(undefined, { hour: 'numeric', minute: '2-digit' });
	}

	/**
	 * Truncate text to N words.
	 */
	function truncateWords(text, numWords) {
		const words = text.split(' ');
		if (words.length <= numWords) {
			return text;
		}
		return words.slice(0, numWords).join(' ') + '...';
	}

	/**
	 * Escape HTML entities.
	 */
	function escapeHtml(text) {
		const div = document.createElement('div');
		div.textContent = text;
		return div.innerHTML;
	}

	/**
	 * Debounce function.
	 */
	function debounce(func, wait) {
		let timeout;
		return function executedFunction(...args) {
			const later = () => {
				clearTimeout(timeout);
				func(...args);
			};
			clearTimeout(timeout);
			timeout = setTimeout(later, wait);
		};
	}

	// Initialize when DOM is ready
	$(document).ready(init);

})(jQuery);
