/**
 * FeedValue Admin JavaScript
 * Enhanced with smooth animations and micro-interactions
 */

(function($) {
	'use strict';

	$(document).ready(function() {
		console.log('FeedValue: Admin JS initialized');
		console.log('FeedValue: Add rule buttons found:', $('.feedvalue-add-rule').length);
		console.log('FeedValue: Preset buttons found:', $('.feedvalue-preset-btn').length);
		console.log('FeedValue: Rule template found:', $('#feedvalue-rule-template').length > 0);
		console.log('FeedValue: Toggle inputs found:', $('.feedvalue-widget-enabled').length);
		console.log('FeedValue: Position selects found:', $('.feedvalue-position-select').length);
		console.log('FeedValue: Widget rows found:', $('.feedvalue-widget-table tbody tr').length);

		/**
		 * Move any notices WordPress injected into header to after it
		 * WordPress auto-injects notices after the first h1 inside .wrap
		 */
		var $header = $('.feedvalue-header');
		if ($header.length) {
			$header.find('.notice, .updated, .error, .settings-error').each(function() {
				$(this).insertAfter($header);
			});
		}

		/**
		 * Test API Connection
		 */
		$('#feedvalue-test-connection').on('click', function(e) {
			e.preventDefault();

			var $button = $(this);
			var $status = $('#feedvalue-connection-status');
			var apiKey = $('#feedvalue_api_key').val();

			// Don't block client-side - server handles empty field by using stored key
			// Server returns appropriate error if no key saved and field is empty

			$button.prop('disabled', true);
			$status.removeClass('success error').addClass('loading')
				.html('<span class="feedvalue-status-text">' + feedvalueAdmin.strings.testing + '</span>')
				.css('display', 'flex');

			$.ajax({
				url: feedvalueAdmin.ajaxurl,
				type: 'POST',
				data: {
					action: 'feedvalue_test_connection',
					nonce: feedvalueAdmin.testConnectionNonce,
					api_key: apiKey  // Server uses stored key if empty
				},
				success: function(response) {
					if (response.success) {
						$status.removeClass('loading error').addClass('success')
							.html('<span class="feedvalue-status-text">' + response.data.message + '</span>');
					} else {
						$status.removeClass('loading success').addClass('error')
							.html('<span class="feedvalue-status-text">' + response.data.message + '</span>');
					}
				},
				error: function() {
					$status.removeClass('loading success').addClass('error')
						.html('<span class="feedvalue-status-text">' + feedvalueAdmin.strings.error + '</span>');
				},
				complete: function() {
					$button.prop('disabled', false);
				}
			});
		});

		/**
		 * Refresh Widgets
		 */
		$('#feedvalue-refresh-widgets').on('click', function(e) {
			e.preventDefault();

			var $button = $(this);
			var $icon = $button.find('.dashicons');
			var originalText = $button.find('.dashicons').next().text() || 'Refresh';

			$button.prop('disabled', true);
			$icon.addClass('spin');

			$.ajax({
				url: feedvalueAdmin.ajaxurl,
				type: 'POST',
				data: {
					action: 'feedvalue_refresh_widgets',
					nonce: feedvalueAdmin.refreshWidgetsNonce
				},
				success: function(response) {
					if (response.success) {
						// Add a brief success state before reload
						$icon.removeClass('spin dashicons-update').addClass('dashicons-yes');
						setTimeout(function() {
							location.reload();
						}, 500);
					} else {
						$icon.removeClass('spin');
						showNotification(response.data.message || feedvalueAdmin.strings.error, 'error');
						$button.prop('disabled', false);
					}
				},
				error: function() {
					$icon.removeClass('spin');
					showNotification(feedvalueAdmin.strings.error, 'error');
					$button.prop('disabled', false);
				}
			});
		});

		/**
		 * Clear Cache (handles both header and inline buttons)
		 */
		$('.feedvalue-clear-cache-btn').on('click', function(e) {
			e.preventDefault();

			if (!confirm(feedvalueAdmin.strings.confirmClearCache)) {
				return;
			}

			var $button = $(this);
			var $icon = $button.find('.dashicons');

			$button.prop('disabled', true);
			$icon.addClass('spin');

			$.ajax({
				url: feedvalueAdmin.ajaxurl,
				type: 'POST',
				data: {
					action: 'feedvalue_clear_cache',
					nonce: feedvalueAdmin.clearCacheNonce
				},
				success: function(response) {
					if (response.success) {
						$icon.removeClass('spin dashicons-trash').addClass('dashicons-yes');
						showNotification(response.data.message, 'success');
						setTimeout(function() {
							location.reload();
						}, 1000);
					} else {
						$icon.removeClass('spin');
						showNotification(response.data.message || feedvalueAdmin.strings.error, 'error');
						$button.prop('disabled', false);
					}
				},
				error: function() {
					$icon.removeClass('spin');
					showNotification(feedvalueAdmin.strings.error, 'error');
					$button.prop('disabled', false);
				}
			});
		});

		/**
		 * Copy Shortcode with enhanced feedback (button)
		 */
		$('.feedvalue-copy-shortcode').on('click', function(e) {
			e.preventDefault();

			var $button = $(this);
			var shortcode = $button.data('shortcode');
			var $icon = $button.find('.dashicons');
			var $text = $button.find('.feedvalue-copy-text');
			var originalIcon = 'dashicons-clipboard';

			// Modern clipboard API with fallback
			if (navigator.clipboard && navigator.clipboard.writeText) {
				navigator.clipboard.writeText(shortcode).then(function() {
					showCopySuccess($button, $icon, $text, originalIcon);
				}).catch(function() {
					fallbackCopy(shortcode, $button, $icon, $text, originalIcon);
				});
			} else {
				fallbackCopy(shortcode, $button, $icon, $text, originalIcon);
			}
		});

		/**
		 * Click-to-copy on shortcode element (truncated display)
		 */
		$('.feedvalue-clickable-copy').on('click', function(e) {
			e.preventDefault();

			var $code = $(this);
			var shortcode = $code.data('shortcode');
			var originalText = $code.text();

			// Modern clipboard API with fallback
			if (navigator.clipboard && navigator.clipboard.writeText) {
				navigator.clipboard.writeText(shortcode).then(function() {
					showCodeCopySuccess($code, originalText);
				}).catch(function() {
					fallbackCodeCopy(shortcode, $code, originalText);
				});
			} else {
				fallbackCodeCopy(shortcode, $code, originalText);
			}
		});

		/**
		 * Fallback copy for code element
		 */
		function fallbackCodeCopy(text, $code, originalText) {
			var $temp = $('<textarea>');
			$('body').append($temp);
			$temp.val(text).select();

			try {
				document.execCommand('copy');
				showCodeCopySuccess($code, originalText);
			} catch (err) {
				showNotification('Failed to copy shortcode', 'error');
			}

			$temp.remove();
		}

		/**
		 * Show copy success for code element
		 */
		function showCodeCopySuccess($code, originalText) {
			$code.addClass('copied').text('Copied!');

			setTimeout(function() {
				$code.removeClass('copied').text(originalText);
			}, 1500);
		}

		/**
		 * Fallback copy method for older browsers
		 */
		function fallbackCopy(text, $button, $icon, $text, originalIcon) {
			var $temp = $('<textarea>');
			$('body').append($temp);
			$temp.val(text).select();

			try {
				document.execCommand('copy');
				showCopySuccess($button, $icon, $text, originalIcon);
			} catch (err) {
				showNotification('Failed to copy shortcode', 'error');
			}

			$temp.remove();
		}

		/**
		 * Show copy success state
		 */
		function showCopySuccess($button, $icon, $text, originalIcon) {
			$button.addClass('copied');
			$icon.removeClass(originalIcon).addClass('dashicons-yes');
			$text.text('Copied!');

			setTimeout(function() {
				$button.removeClass('copied');
				$icon.removeClass('dashicons-yes').addClass(originalIcon);
				$text.text('Copy');
			}, 2000);
		}

		/**
		 * Targeting System - Rule Builder
		 */
		var targetingRuleIndex = 0;

		/**
		 * Configure Targeting Rules (Modal)
		 */
		$('.feedvalue-configure-targeting').on('click', function(e) {
			e.preventDefault();

			var widgetId = $(this).data('widget-id');
			var widgetName = $(this).data('widget-name');

			var $modal = $('#feedvalue-targeting-modal');
			$modal.data('widget-id', widgetId);
			$('#feedvalue-targeting-widget-name').text(widgetName);

			// Reset modal state
			resetTargetingModal();

			// Load existing rules
			loadTargetingRules(widgetId);

			$modal.fadeIn(200);
			$('body').css('overflow', 'hidden');
		});

		/**
		 * Reset targeting modal to initial state
		 */
		function resetTargetingModal() {
			targetingRuleIndex = 0;
			$('#feedvalue-inclusion-rules').html(
				'<div class="feedvalue-no-rules">' +
				'<span class="dashicons dashicons-info-outline"></span>' +
				'No inclusion rules. Widget will show on all pages (unless excluded below).' +
				'</div>'
			);
			$('#feedvalue-exclusion-rules').html(
				'<div class="feedvalue-no-rules">' +
				'<span class="dashicons dashicons-info-outline"></span>' +
				'No exclusion rules. Widget will not be hidden from any pages.' +
				'</div>'
			);
		}

		/**
		 * Load existing targeting rules for a widget
		 */
		function loadTargetingRules(widgetId) {
			$.ajax({
				url: feedvalueAdmin.ajaxurl,
				type: 'POST',
				data: {
					action: 'feedvalue_get_targeting_rules',
					nonce: feedvalueAdmin.saveTargetingNonce,
					widget_id: widgetId
				},
				success: function(response) {
					if (response.success && response.data.rules) {
						var rules = response.data.rules;

						// Load inclusion rules
						if (rules.inclusion && rules.inclusion.length > 0) {
							$('#feedvalue-inclusion-rules').empty();
							rules.inclusion.forEach(function(rule) {
								addRuleCard('inclusion', rule);
							});
						}

						// Load exclusion rules
						if (rules.exclusion && rules.exclusion.length > 0) {
							$('#feedvalue-exclusion-rules').empty();
							rules.exclusion.forEach(function(rule) {
								addRuleCard('exclusion', rule);
							});
						}
					}
				}
			});
		}

		/**
		 * Add Rule Button Handler
		 */
		$(document).on('click', '.feedvalue-add-rule', function(e) {
			e.preventDefault();
			e.stopPropagation();
			// jQuery converts data-rule-type to ruleType (camelCase)
			var ruleType = $(this).attr('data-rule-type');
			console.log('FeedValue: Add Rule clicked, type:', ruleType);
			if (!ruleType) {
				console.error('FeedValue: No rule type found on button');
				return;
			}
			addRuleCard(ruleType);
		});

		/**
		 * Add a rule card to the specified list
		 */
		function addRuleCard(listType, existingRule) {
			console.log('FeedValue: addRuleCard called with listType:', listType);
			var $list = $('#feedvalue-' + listType + '-rules');

			if ($list.length === 0) {
				console.error('FeedValue: Could not find rules list for type:', listType);
				return;
			}

			// Remove "no rules" message if present
			$list.find('.feedvalue-no-rules').remove();

			// Get template and create new card
			var template = $('#feedvalue-rule-template').html();
			console.log('FeedValue: Template found:', template ? 'yes' : 'no');

			if (!template) {
				console.error('FeedValue: Could not find rule template');
				return;
			}

			var $card = $(template.replace('{index}', targetingRuleIndex));
			targetingRuleIndex++;

			$list.append($card);
			console.log('FeedValue: Rule card added');

			// If we have an existing rule, set its values
			if (existingRule) {
				var $select = $card.find('.feedvalue-rule-type');
				$select.val(existingRule.type);

				// Trigger change to show value field if needed
				$select.trigger('change');

				// Set the value after the field is shown
				setTimeout(function() {
					setRuleValue($card, existingRule);
				}, 50);
			}
		}

		/**
		 * Set rule value from existing data
		 */
		function setRuleValue($card, rule) {
			var $valueContainer = $card.find('.feedvalue-rule-value');

			switch (rule.type) {
				case 'post_type':
				case 'categories':
				case 'tags':
					// Checkbox values
					var values = Array.isArray(rule.value) ? rule.value : [rule.value];
					values.forEach(function(val) {
						$valueContainer.find('input[value="' + val + '"]').prop('checked', true);
					});
					break;

				case 'post_ids':
				case 'url_pattern':
					// Text input values
					$valueContainer.find('.feedvalue-text-input').val(rule.value);
					break;
			}
		}

		/**
		 * Remove Rule Button Handler
		 */
		$(document).on('click', '.feedvalue-remove-rule', function(e) {
			e.preventDefault();
			var $card = $(this).closest('.feedvalue-rule-card');
			var $list = $card.parent();

			$card.css('opacity', '0').slideUp(200, function() {
				$(this).remove();

				// If no more rules, show the "no rules" message
				if ($list.children('.feedvalue-rule-card').length === 0) {
					var listId = $list.attr('id');
					var message = listId.indexOf('inclusion') !== -1
						? 'No inclusion rules. Widget will show on all pages (unless excluded below).'
						: 'No exclusion rules. Widget will not be hidden from any pages.';

					$list.html(
						'<div class="feedvalue-no-rules">' +
						'<span class="dashicons dashicons-info-outline"></span>' +
						message +
						'</div>'
					);
				}
			});
		});

		/**
		 * Rule Type Change Handler
		 */
		$(document).on('change', '.feedvalue-rule-type', function() {
			var $select = $(this);
			var $card = $select.closest('.feedvalue-rule-card');
			var $valueContainer = $card.find('.feedvalue-rule-value');
			var ruleType = $select.val();

			// Get the template based on rule type
			var templateId = null;
			switch (ruleType) {
				case 'post_type':
					templateId = 'feedvalue-post-type-template';
					break;
				case 'post_ids':
					templateId = 'feedvalue-post-ids-template';
					break;
				case 'categories':
					templateId = 'feedvalue-categories-template';
					break;
				case 'tags':
					templateId = 'feedvalue-tags-template';
					break;
				case 'url_pattern':
					templateId = 'feedvalue-url-pattern-template';
					break;
			}

			if (templateId) {
				var template = $('#' + templateId).html();
				$valueContainer.html(template).slideDown(200);
			} else {
				$valueContainer.slideUp(200, function() {
					$(this).empty();
				});
			}
		});

		/**
		 * Preset Button Handlers
		 */
		$(document).on('click', '.feedvalue-preset-btn', function(e) {
			e.preventDefault();
			e.stopPropagation();
			var preset = $(this).data('preset');
			console.log('FeedValue: Preset clicked:', preset);

			if (!preset) {
				console.error('FeedValue: No preset found on button');
				return;
			}

			// Clear existing rules
			resetTargetingModal();

			switch (preset) {
				case 'all-pages':
					addRuleCard('inclusion', { type: 'all_pages' });
					break;

				case 'blog-only':
					addRuleCard('inclusion', { type: 'post_type', value: ['post'] });
					break;

				case 'homepage-only':
					addRuleCard('inclusion', { type: 'homepage' });
					break;

				case 'not-homepage':
					addRuleCard('inclusion', { type: 'all_pages' });
					addRuleCard('exclusion', { type: 'homepage' });
					break;

				default:
					console.warn('FeedValue: Unknown preset:', preset);
			}

			showNotification('Preset applied. Don\'t forget to save!', 'info');
		});

		/**
		 * Close Modal
		 */
		$('.feedvalue-modal-close').on('click', function(e) {
			e.preventDefault();
			closeModal();
		});

		/**
		 * Click outside modal to close
		 */
		$('.feedvalue-modal').on('click', function(e) {
			if ($(e.target).hasClass('feedvalue-modal')) {
				closeModal();
			}
		});

		/**
		 * Escape key to close modal
		 */
		$(document).on('keydown', function(e) {
			if (e.key === 'Escape' && $('.feedvalue-modal:visible').length) {
				closeModal();
			}
		});

		/**
		 * Close modal helper
		 */
		function closeModal() {
			$('.feedvalue-modal').fadeOut(200);
			$('body').css('overflow', '');
		}

		/**
		 * Collect rules from the modal
		 */
		function collectTargetingRules() {
			var rules = {
				inclusion: [],
				exclusion: []
			};

			// Collect inclusion rules
			$('#feedvalue-inclusion-rules .feedvalue-rule-card').each(function() {
				var rule = collectRuleFromCard($(this));
				if (rule) {
					rules.inclusion.push(rule);
				}
			});

			// Collect exclusion rules
			$('#feedvalue-exclusion-rules .feedvalue-rule-card').each(function() {
				var rule = collectRuleFromCard($(this));
				if (rule) {
					rules.exclusion.push(rule);
				}
			});

			return rules;
		}

		/**
		 * Collect a single rule from a card
		 */
		function collectRuleFromCard($card) {
			var type = $card.find('.feedvalue-rule-type').val();
			if (!type) {
				return null;
			}

			var rule = { type: type };

			// Get value based on rule type
			switch (type) {
				case 'post_type':
				case 'categories':
				case 'tags':
					// Collect checked checkboxes
					var values = [];
					$card.find('.feedvalue-rule-value input[type="checkbox"]:checked').each(function() {
						values.push($(this).val());
					});
					if (values.length > 0) {
						rule.value = values;
					}
					break;

				case 'post_ids':
				case 'url_pattern':
					// Get text input value
					var value = $card.find('.feedvalue-text-input').val();
					if (value) {
						rule.value = value;
					}
					break;
			}

			return rule;
		}

		/**
		 * Save Targeting Rules
		 */
		$('.feedvalue-save-targeting').on('click', function(e) {
			e.preventDefault();

			var $button = $(this);
			var $modal = $('#feedvalue-targeting-modal');
			var widgetId = $modal.data('widget-id');
			var $icon = $button.find('.dashicons');
			var originalIconClass = 'dashicons-saved';

			// Collect rules from the UI
			var rules = collectTargetingRules();

			$button.prop('disabled', true);
			$icon.removeClass(originalIconClass).addClass('dashicons-update spin');

			$.ajax({
				url: feedvalueAdmin.ajaxurl,
				type: 'POST',
				data: {
					action: 'feedvalue_save_targeting_rules',
					nonce: feedvalueAdmin.saveTargetingNonce,
					widget_id: widgetId,
					rules: JSON.stringify(rules)
				},
				success: function(response) {
					if (response.success) {
						$icon.removeClass('dashicons-update spin').addClass('dashicons-yes-alt');
						showNotification(response.data.message, 'success');
						setTimeout(function() {
							closeModal();
							$icon.removeClass('dashicons-yes-alt').addClass(originalIconClass);
						}, 800);
					} else {
						$icon.removeClass('dashicons-update spin').addClass(originalIconClass);
						showNotification(response.data.message || feedvalueAdmin.strings.error, 'error');
					}
				},
				error: function() {
					$icon.removeClass('dashicons-update spin').addClass(originalIconClass);
					showNotification(feedvalueAdmin.strings.error, 'error');
				},
				complete: function() {
					$button.prop('disabled', false);
				}
			});
		});

		/**
		 * Save widget settings via AJAX
		 */
		function saveWidgetSettings(widgetId, enabled, globalPosition, $row) {
			console.log('FeedValue: Saving widget settings', {
				widget_id: widgetId,
				enabled: enabled,
				global_position: globalPosition
			});

			if (!widgetId) {
				console.error('FeedValue: No widget ID found');
				showNotification('Error: No widget ID', 'error');
				return;
			}

			// Visual feedback
			$row.css('background-color', 'var(--fv-primary-50)');

			$.ajax({
				url: feedvalueAdmin.ajaxurl,
				type: 'POST',
				data: {
					action: 'feedvalue_save_widget_settings',
					nonce: feedvalueAdmin.saveWidgetSettingsNonce,
					widget_id: widgetId,
					enabled: enabled,
					global_position: globalPosition
				},
				success: function(response) {
					console.log('FeedValue: Save response', response);
					if (response.success) {
						showNotification(feedvalueAdmin.strings.saved, 'success');
					} else {
						showNotification(response.data.message || feedvalueAdmin.strings.error, 'error');
					}
				},
				error: function(xhr, status, error) {
					console.error('FeedValue: AJAX error', { xhr: xhr, status: status, error: error });
					showNotification(feedvalueAdmin.strings.error, 'error');
				},
				complete: function() {
					setTimeout(function() {
						$row.css('background-color', '');
					}, 300);
				}
			});
		}

		/**
		 * Toggle widget enabled state with auto-save
		 */
		$(document).on('change', '.feedvalue-widget-enabled', function() {
			var $toggle = $(this);
			var $row = $toggle.closest('tr');
			var widgetId = $row.attr('data-widget-id');
			var enabled = $toggle.is(':checked');
			var globalPosition = $row.find('.feedvalue-position-select').val() || '';

			console.log('FeedValue: Toggle changed', { widgetId: widgetId, enabled: enabled });
			saveWidgetSettings(widgetId, enabled, globalPosition, $row);
		});

		/**
		 * Auto-Display position change with auto-save
		 */
		$(document).on('change', '.feedvalue-position-select', function() {
			var $select = $(this);
			var $row = $select.closest('tr');
			var widgetId = $row.attr('data-widget-id');
			var enabled = $row.find('.feedvalue-widget-enabled').is(':checked');
			var globalPosition = $select.val();

			console.log('FeedValue: Position changed', { widgetId: widgetId, globalPosition: globalPosition });
			saveWidgetSettings(widgetId, enabled, globalPosition, $row);
		});

		/**
		 * Show notification toast
		 */
		function showNotification(message, type) {
			// Remove existing notifications
			$('.feedvalue-notification').remove();

			var bgColor = type === 'success' ? 'var(--fv-success-500)' :
						  type === 'error' ? 'var(--fv-error-500)' : 'var(--fv-info-500)';

			var $notification = $('<div class="feedvalue-notification">')
				.css({
					position: 'fixed',
					bottom: '24px',
					right: '24px',
					background: bgColor,
					color: 'white',
					padding: '14px 24px',
					borderRadius: '10px',
					boxShadow: '0 4px 20px rgba(0, 0, 0, 0.2)',
					zIndex: 100001,
					fontFamily: "'DM Sans', sans-serif",
					fontSize: '14px',
					fontWeight: '500',
					opacity: 0,
					transform: 'translateY(20px)'
				})
				.text(message)
				.appendTo('body');

			// Animate in
			$notification.animate({
				opacity: 1,
				bottom: '32px'
			}, 300);

			// Remove after delay
			setTimeout(function() {
				$notification.animate({
					opacity: 0,
					bottom: '24px'
				}, 300, function() {
					$(this).remove();
				});
			}, 3000);
		}

		/**
		 * Add row hover effects
		 */
		$('.feedvalue-widget-table tbody tr').on('mouseenter', function() {
			$(this).find('.feedvalue-toggle-slider').css('transform', 'scale(1.02)');
		}).on('mouseleave', function() {
			$(this).find('.feedvalue-toggle-slider').css('transform', 'scale(1)');
		});

		/**
		 * Make Advanced Settings collapsible
		 */
		(function initCollapsibleSections() {
			// Find the Advanced Settings section by its heading text
			$('.feedvalue-settings-section h2').each(function() {
				var $heading = $(this);
				var headingText = $heading.text().trim();

				if (headingText === 'Advanced Settings') {
					var $table = $heading.next('p').next('.form-table');
					if ($table.length) {
						// Wrap the content
						$heading.add($heading.next('p')).add($table).wrapAll('<div class="feedvalue-collapsible collapsed"></div>');
						var $collapsible = $heading.closest('.feedvalue-collapsible');

						// Add header wrapper with icon
						$heading.wrap('<div class="feedvalue-collapsible-header"></div>');
						$heading.after('<span class="dashicons dashicons-arrow-down-alt2 feedvalue-collapse-icon"></span>');

						// Wrap content
						$heading.parent().nextAll().wrapAll('<div class="feedvalue-collapsible-content"></div>');

						// Toggle on click
						$collapsible.find('.feedvalue-collapsible-header').on('click', function() {
							$collapsible.toggleClass('collapsed');
						});
					}
				}
			});
		})();

		/**
		 * Feedback page functionality
		 */
		if ($('.feedvalue-feedback-page').length) {
			// Select all checkbox
			$('#feedvalue-select-all').on('change', function() {
				var isChecked = $(this).prop('checked');
				$('.feedvalue-feedback-checkbox').prop('checked', isChecked);
				updateSelectedCount();
			});

			// Individual checkboxes
			$('.feedvalue-feedback-checkbox').on('change', function() {
				updateSelectedCount();
				// Update select all state
				var totalCheckboxes = $('.feedvalue-feedback-checkbox').length;
				var checkedCheckboxes = $('.feedvalue-feedback-checkbox:checked').length;
				$('#feedvalue-select-all').prop('checked', totalCheckboxes === checkedCheckboxes);
			});

			function updateSelectedCount() {
				var count = $('.feedvalue-feedback-checkbox:checked').length;
				if (count > 0) {
					$('.feedvalue-selected-count').text(count + ' selected');
				} else {
					$('.feedvalue-selected-count').text('');
				}
			}

			// Expand feedback text
			$('.feedvalue-expand-feedback').on('click', function() {
				var $btn = $(this);
				var $textDiv = $btn.siblings('.feedvalue-feedback-text');
				var fullText = $textDiv.data('full-text');
				var isExpanded = $btn.data('expanded');

				if (isExpanded) {
					// Collapse - show truncated
					var words = fullText.split(' ');
					var truncated = words.slice(0, 15).join(' ') + '...';
					$textDiv.text(truncated);
					$btn.text('Read more');
					$btn.data('expanded', false);
				} else {
					// Expand - show full text
					$textDiv.text(fullText);
					$btn.text('Show less');
					$btn.data('expanded', true);
				}
			});

			// Mark as reviewed (demo)
			$('.feedvalue-mark-reviewed').on('click', function() {
				var $row = $(this).closest('tr');
				$row.find('.feedvalue-badge').removeClass('feedvalue-badge-new').addClass('feedvalue-badge-reviewed').text('Reviewed');
				$(this).fadeOut();
				showNotification('Marked as reviewed', 'success');
			});

			// Archive feedback (demo)
			$('.feedvalue-archive-feedback').on('click', function() {
				var $row = $(this).closest('tr');
				$row.find('.feedvalue-badge').removeClass('feedvalue-badge-new feedvalue-badge-reviewed').addClass('feedvalue-badge-archived').text('Archived');
				$row.css('opacity', '0.6');
				showNotification('Feedback archived', 'success');
			});

			// Export CSV (demo)
			$('#feedvalue-export-feedback').on('click', function() {
				showNotification('Export feature coming soon!', 'info');
			});

			// Apply bulk action (demo)
			$('#feedvalue-apply-bulk').on('click', function() {
				var action = $('#feedvalue-bulk-action').val();
				var selected = $('.feedvalue-feedback-checkbox:checked').length;

				if (!action) {
					showNotification('Please select an action', 'error');
					return;
				}

				if (selected === 0) {
					showNotification('Please select at least one item', 'error');
					return;
				}

				showNotification('Bulk action "' + action + '" applied to ' + selected + ' items', 'success');
			});

			// Filter change handlers (demo)
			$('.feedvalue-filter-select, .feedvalue-search-input').on('change keyup', function() {
				// In real implementation, this would filter the table
				// For demo, just show a notification
				if ($(this).is('.feedvalue-search-input') && $(this).val().length > 2) {
					showNotification('Searching...', 'info');
				}
			});
		}

		/**
		 * Show notification helper (reuse existing function pattern)
		 */
		function showNotification(message, type) {
			// Remove existing notifications
			$('.feedvalue-notification').remove();

			var bgColor = type === 'success' ? 'var(--fv-success-500)' :
						  type === 'error' ? 'var(--fv-error-500)' : 'var(--fv-info-500)';

			var $notification = $('<div class="feedvalue-notification">')
				.css({
					position: 'fixed',
					bottom: '24px',
					right: '24px',
					background: bgColor,
					color: 'white',
					padding: '14px 24px',
					borderRadius: '10px',
					boxShadow: '0 4px 20px rgba(0, 0, 0, 0.2)',
					zIndex: 100001,
					fontFamily: "'DM Sans', sans-serif",
					fontSize: '14px',
					fontWeight: '500',
					opacity: 0,
					transform: 'translateY(20px)'
				})
				.text(message)
				.appendTo('body');

			// Animate in
			$notification.animate({
				opacity: 1,
				bottom: '32px'
			}, 300);

			// Remove after delay
			setTimeout(function() {
				$notification.animate({
					opacity: 0,
					bottom: '24px'
				}, 300, function() {
					$(this).remove();
				});
			}, 3000);
		}

		/**
		 * Shortcode Generator Modal
		 */
		var currentShortcodeWidgetId = '';

		/**
		 * Open shortcode generator modal
		 */
		$('.feedvalue-shortcode-generator-btn').on('click', function(e) {
			e.preventDefault();

			var widgetId = $(this).data('widget-id');
			var widgetName = $(this).data('widget-name');

			currentShortcodeWidgetId = widgetId;

			var $modal = $('#feedvalue-shortcode-modal');
			$('#feedvalue-shortcode-widget-name').text(widgetName);

			// Reset options
			$('#feedvalue-shortcode-class').val('');
			$('#feedvalue-shortcode-id').val('');

			// Generate initial shortcode
			updateGeneratedShortcode();

			$modal.fadeIn(200);
			$('body').css('overflow', 'hidden');
		});

		/**
		 * Update generated shortcode based on options
		 */
		function updateGeneratedShortcode() {
			if (!currentShortcodeWidgetId) return;

			var shortcode = '[feedvalue widget_id="' + currentShortcodeWidgetId + '"';

			var cssClass = $('#feedvalue-shortcode-class').val().trim();
			var cssId = $('#feedvalue-shortcode-id').val().trim();

			if (cssClass) {
				// Sanitize class names (allow alphanumeric, dashes, underscores, spaces)
				cssClass = cssClass.replace(/[^a-zA-Z0-9_\-\s]/g, '');
				if (cssClass) {
					shortcode += ' class="' + cssClass + '"';
				}
			}

			if (cssId) {
				// Sanitize ID (allow alphanumeric, dashes, underscores)
				cssId = cssId.replace(/[^a-zA-Z0-9_\-]/g, '');
				if (cssId) {
					shortcode += ' id="' + cssId + '"';
				}
			}

			shortcode += ']';

			// Update the shortcode display
			$('#feedvalue-generated-shortcode').text(shortcode);

			// Update the PHP code display
			var phpCode = '<?php echo do_shortcode(\'' + shortcode.replace(/'/g, "\\'") + '\'); ?>';
			$('#feedvalue-generated-php').text(phpCode);
		}

		/**
		 * Update shortcode when options change
		 */
		$(document).on('input', '.feedvalue-shortcode-option', function() {
			updateGeneratedShortcode();
		});

		/**
		 * Copy generated shortcode
		 */
		$('#feedvalue-copy-generated').on('click', function(e) {
			e.preventDefault();
			var shortcode = $('#feedvalue-generated-shortcode').text();
			copyToClipboard(shortcode, $(this));
		});

		/**
		 * Copy PHP code
		 */
		$('#feedvalue-copy-php').on('click', function(e) {
			e.preventDefault();
			var phpCode = $('#feedvalue-generated-php').text();
			copyToClipboard(phpCode, $(this));
		});

		/**
		 * Helper to copy text to clipboard with button feedback
		 */
		function copyToClipboard(text, $button) {
			var $icon = $button.find('.dashicons');
			var originalIconClass = 'dashicons-clipboard';

			if (navigator.clipboard && navigator.clipboard.writeText) {
				navigator.clipboard.writeText(text).then(function() {
					showCopyButtonSuccess($button, $icon, originalIconClass);
				}).catch(function() {
					fallbackCopyToClipboard(text, $button, $icon, originalIconClass);
				});
			} else {
				fallbackCopyToClipboard(text, $button, $icon, originalIconClass);
			}
		}

		/**
		 * Fallback copy method
		 */
		function fallbackCopyToClipboard(text, $button, $icon, originalIconClass) {
			var $temp = $('<textarea>');
			$('body').append($temp);
			$temp.val(text).select();

			try {
				document.execCommand('copy');
				showCopyButtonSuccess($button, $icon, originalIconClass);
			} catch (err) {
				showNotification('Failed to copy', 'error');
			}

			$temp.remove();
		}

		/**
		 * Show copy button success state
		 */
		function showCopyButtonSuccess($button, $icon, originalIconClass) {
			$icon.removeClass(originalIconClass).addClass('dashicons-yes');
			$button.addClass('copied');

			setTimeout(function() {
				$icon.removeClass('dashicons-yes').addClass(originalIconClass);
				$button.removeClass('copied');
			}, 1500);

			showNotification('Copied!', 'success');
		}

	});

})(jQuery);
