<?php
/**
 * Settings API functionality.
 *
 * @package FeedValue
 */

class FeedValue_Settings {

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		// Register settings
		register_setting(
			'feedvalue_settings_group',
			'feedvalue_api_key',
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_api_key' ),
				'default'           => '',
			)
		);

		register_setting(
			'feedvalue_settings_group',
			'feedvalue_cache_ttl',
			array(
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => FEEDVALUE_CACHE_TTL,
			)
		);

		// NOTE: feedvalue_widget_settings is NOT registered here because it's managed
		// via AJAX (save_widget_settings). Registering it in the settings group would
		// cause it to be reset to default when the API Configuration form is saved.

		register_setting(
			'feedvalue_settings_group',
			'feedvalue_api_url_override',
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_api_url' ),
				'default'           => '',
			)
		);

		// Add settings sections
		add_settings_section(
			'feedvalue_api_section',
			__( 'FeedValue API Configuration', 'feedvalue' ),
			array( $this, 'render_api_section' ),
			'feedvalue'
		);

		// Widget Management section removed - now handled in widget list template

		add_settings_section(
			'feedvalue_advanced_section',
			__( 'Advanced Settings', 'feedvalue' ),
			array( $this, 'render_advanced_section' ),
			'feedvalue'
		);

		add_settings_section(
			'feedvalue_gdpr_section',
			__( 'Privacy & GDPR Compliance', 'feedvalue' ),
			array( $this, 'render_gdpr_section' ),
			'feedvalue'
		);

		// Add settings fields
		add_settings_field(
			'feedvalue_api_key',
			__( 'FeedValue API Key', 'feedvalue' ),
			array( $this, 'render_api_key_field' ),
			'feedvalue',
			'feedvalue_api_section'
		);

		add_settings_field(
			'feedvalue_cache_ttl',
			__( 'Cache Duration (seconds)', 'feedvalue' ),
			array( $this, 'render_cache_ttl_field' ),
			'feedvalue',
			'feedvalue_api_section'
		);

		add_settings_field(
			'feedvalue_api_url_override',
			__( 'API URL Override', 'feedvalue' ),
			array( $this, 'render_api_url_field' ),
			'feedvalue',
			'feedvalue_advanced_section'
		);
	}

	/**
	 * Sanitize API key input.
	 *
	 * @param string $api_key API key input.
	 * @return string Sanitized API key.
	 */
	public function sanitize_api_key( $api_key ) {
		$api_key = sanitize_text_field( $api_key );

		// If empty, keep the existing stored key (don't overwrite on other settings saves)
		if ( empty( $api_key ) ) {
			return get_option( 'feedvalue_api_key', '' );
		}

		// Encrypt API key before storing (use WordPress salts)
		$api_key = $this->encrypt_api_key( $api_key );

		// Test connection when API key is saved
		if ( ! empty( $api_key ) ) {
			$decrypted_key = $this->decrypt_api_key( $api_key );
			$api_client    = new FeedValue_API_Client();
			$result        = $api_client->test_connection( $decrypted_key );

			if ( is_wp_error( $result ) ) {
				add_settings_error(
					'feedvalue_api_key',
					'invalid_api_key',
					sprintf(
						/* translators: %s: error message */
						__( 'API Key Error: %s', 'feedvalue' ),
						$result->get_error_message()
					),
					'error'
				);
			} else {
				add_settings_error(
					'feedvalue_api_key',
					'connection_success',
					__( 'Connection successful! Your widgets will be loaded automatically.', 'feedvalue' ),
					'success'
				);

				// Clear cache to force refresh
				$cache = new FeedValue_Cache();
				$cache->clear_all();
			}
		}

		do_action( 'feedvalue_settings_saved', array( 'api_key' => $api_key ) );

		return $api_key;
	}

	/**
	 * Sanitize widget settings.
	 *
	 * @param array $settings Widget settings.
	 * @return array Sanitized settings.
	 */
	public function sanitize_widget_settings( $settings ) {
		if ( ! is_array( $settings ) ) {
			return array();
		}

		$sanitized = array();

		foreach ( $settings as $widget_id => $widget_settings ) {
			$widget_id = sanitize_text_field( $widget_id );

			$sanitized[ $widget_id ] = array(
				'enabled'         => isset( $widget_settings['enabled'] ) ? (bool) $widget_settings['enabled'] : false,
				'global_position' => isset( $widget_settings['global_position'] ) ? sanitize_text_field( $widget_settings['global_position'] ) : '',
			);
		}

		return $sanitized;
	}

	/**
	 * Encrypt API key for storage.
	 *
	 * @param string $api_key Plain text API key.
	 * @return string Encrypted API key.
	 */
	private function encrypt_api_key( $api_key ) {
		if ( ! function_exists( 'openssl_encrypt' ) ) {
			// Fallback: base64 encode (not secure, but better than nothing)
			return base64_encode( $api_key ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
		}

		$key    = wp_salt( 'auth' );
		$iv     = substr( wp_salt( 'secure_auth' ), 0, 16 );
		$cipher = 'AES-256-CBC';

		$encrypted = openssl_encrypt( $api_key, $cipher, $key, 0, $iv );

		return base64_encode( $encrypted ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
	}

	/**
	 * Decrypt API key from storage.
	 *
	 * @param string $encrypted_key Encrypted API key.
	 * @return string Decrypted API key.
	 */
	public function decrypt_api_key( $encrypted_key ) {
		if ( ! function_exists( 'openssl_decrypt' ) ) {
			// Fallback: base64 decode
			return base64_decode( $encrypted_key ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
		}

		$key       = wp_salt( 'auth' );
		$iv        = substr( wp_salt( 'secure_auth' ), 0, 16 );
		$cipher    = 'AES-256-CBC';
		$encrypted = base64_decode( $encrypted_key ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode

		return openssl_decrypt( $encrypted, $cipher, $key, 0, $iv );
	}

	/**
	 * Render API section description.
	 */
	public function render_api_section() {
		?>
		<p>
			<?php
			printf(
				/* translators: %s: FeedValue dashboard URL */
				esc_html__( 'Enter your FeedValue API key from your %s.', 'feedvalue' ),
				'<a href="https://feedvalue.com/app" target="_blank" rel="noopener noreferrer">' . esc_html__( 'FeedValue dashboard', 'feedvalue' ) . '</a>'
			);
			?>
		</p>
		<?php
	}

	/**
	 * Render API key field.
	 */
	public function render_api_key_field() {
		$encrypted_key = get_option( 'feedvalue_api_key', '' );
		$has_key       = ! empty( $encrypted_key );
		?>
		<input
			type="password"
			name="feedvalue_api_key"
			id="feedvalue_api_key"
			value=""
			class="regular-text"
			placeholder="<?php echo $has_key ? esc_attr__( 'API key saved (leave blank to keep)', 'feedvalue' ) : esc_attr__( 'Enter your API key', 'feedvalue' ); ?>"
		/>
		<button type="button" id="feedvalue-test-connection" class="button button-secondary">
			<?php esc_html_e( 'Test Connection', 'feedvalue' ); ?>
		</button>
		<?php if ( $has_key ) : ?>
			<span class="feedvalue-key-status feedvalue-key-saved">
				<span class="dashicons dashicons-yes-alt"></span>
				<?php esc_html_e( 'Key saved', 'feedvalue' ); ?>
			</span>
		<?php endif; ?>
		<p class="description">
			<?php
			printf(
				/* translators: %s: FeedValue dashboard URL */
				esc_html__( 'Find your API key in your %s.', 'feedvalue' ),
				'<a href="https://feedvalue.com/app/settings/api" target="_blank" rel="noopener noreferrer">' . esc_html__( 'FeedValue dashboard', 'feedvalue' ) . '</a>'
			);
			if ( $has_key ) {
				echo ' ';
				esc_html_e( 'Leave blank to keep existing key, or enter a new key to replace it.', 'feedvalue' );
			}
			?>
		</p>
		<div id="feedvalue-connection-status" style="margin-top: 10px;"></div>
		<?php
	}

	/**
	 * Render cache TTL field.
	 */
	public function render_cache_ttl_field() {
		$cache_ttl = get_option( 'feedvalue_cache_ttl', FEEDVALUE_CACHE_TTL );
		?>
		<input
			type="number"
			name="feedvalue_cache_ttl"
			id="feedvalue_cache_ttl"
			value="<?php echo esc_attr( $cache_ttl ); ?>"
			min="60"
			max="3600"
			step="60"
			class="small-text"
		/>
		<button type="button" id="feedvalue-clear-cache-inline" class="button button-secondary feedvalue-clear-cache-btn">
			<?php esc_html_e( 'Clear Cache Now', 'feedvalue' ); ?>
		</button>
		<p class="description">
			<?php esc_html_e( 'How long to cache widget configuration. Default: 300 seconds (5 minutes).', 'feedvalue' ); ?>
		</p>
		<?php
	}

	/**
	 * Render advanced section description.
	 */
	public function render_advanced_section() {
		?>
		<p><?php esc_html_e( 'Advanced configuration options for developers and testing.', 'feedvalue' ); ?></p>
		<?php
	}

	/**
	 * Render GDPR section with privacy information.
	 */
	public function render_gdpr_section() {
		?>
		<div class="feedvalue-gdpr-notice">
			<div class="feedvalue-gdpr-icon">
				<span class="dashicons dashicons-shield"></span>
			</div>
			<div class="feedvalue-gdpr-content">
				<h4><?php esc_html_e( 'Data Processing Information', 'feedvalue' ); ?></h4>
				<p>
					<?php esc_html_e( 'FeedValue collects and processes the following types of data:', 'feedvalue' ); ?>
				</p>
				<ul>
					<li><?php esc_html_e( 'User feedback and responses submitted through widgets', 'feedvalue' ); ?></li>
					<li><?php esc_html_e( 'Anonymous widget interaction analytics (clicks, impressions)', 'feedvalue' ); ?></li>
					<li><?php esc_html_e( 'IP addresses (for rate limiting and fraud prevention)', 'feedvalue' ); ?></li>
					<li><?php esc_html_e( 'Browser/device information (for widget rendering optimization)', 'feedvalue' ); ?></li>
				</ul>
				<p>
					<strong><?php esc_html_e( 'Your Responsibilities:', 'feedvalue' ); ?></strong>
				</p>
				<ul>
					<li><?php esc_html_e( 'Update your Privacy Policy to disclose FeedValue data collection', 'feedvalue' ); ?></li>
					<li><?php esc_html_e( 'Ensure appropriate consent mechanisms are in place (e.g., cookie consent)', 'feedvalue' ); ?></li>
					<li><?php esc_html_e( 'Respond to data subject access requests that may include FeedValue data', 'feedvalue' ); ?></li>
				</ul>
				<p>
					<?php
					printf(
						/* translators: 1: Privacy Policy link, 2: Data Processing Agreement link */
						esc_html__( 'For more information, see our %1$s and %2$s.', 'feedvalue' ),
						'<a href="https://feedvalue.com/privacy" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Privacy Policy', 'feedvalue' ) . '</a>',
						'<a href="https://feedvalue.com/dpa" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Data Processing Agreement', 'feedvalue' ) . '</a>'
					);
					?>
				</p>
				<p class="feedvalue-gdpr-export">
					<span class="dashicons dashicons-download"></span>
					<?php
					printf(
						/* translators: %s: WordPress export tool link */
						esc_html__( 'FeedValue integrates with WordPress personal data export/erasure tools. See %s.', 'feedvalue' ),
						'<a href="' . esc_url( admin_url( 'export-personal-data.php' ) ) . '">' . esc_html__( 'Export Personal Data', 'feedvalue' ) . '</a>'
					);
					?>
				</p>
			</div>
		</div>
		<style>
			.feedvalue-gdpr-notice {
				background: #f0f6fc;
				border: 1px solid #c3c4c7;
				border-left: 4px solid #2271b1;
				padding: 16px;
				display: flex;
				gap: 16px;
				margin-bottom: 20px;
			}
			.feedvalue-gdpr-icon {
				flex-shrink: 0;
			}
			.feedvalue-gdpr-icon .dashicons {
				font-size: 32px;
				width: 32px;
				height: 32px;
				color: #2271b1;
			}
			.feedvalue-gdpr-content h4 {
				margin: 0 0 10px 0;
				font-size: 14px;
			}
			.feedvalue-gdpr-content p {
				margin: 0 0 10px 0;
			}
			.feedvalue-gdpr-content ul {
				margin: 0 0 10px 20px;
				list-style-type: disc;
			}
			.feedvalue-gdpr-content li {
				margin-bottom: 4px;
			}
			.feedvalue-gdpr-export {
				margin-top: 16px !important;
				padding-top: 12px;
				border-top: 1px solid #c3c4c7;
			}
			.feedvalue-gdpr-export .dashicons {
				vertical-align: middle;
				margin-right: 4px;
			}
		</style>
		<?php
	}

	/**
	 * Render API URL override field.
	 */
	public function render_api_url_field() {
		$api_url = get_option( 'feedvalue_api_url_override', '' );
		?>
		<input
			type="url"
			name="feedvalue_api_url_override"
			id="feedvalue_api_url_override"
			value="<?php echo esc_attr( $api_url ); ?>"
			class="regular-text"
			placeholder="<?php echo esc_attr( FEEDVALUE_API_ENDPOINT ); ?>"
		/>
		<p class="description">
			<?php esc_html_e( 'For enterprise customers with a dedicated API instance. Leave blank to use the default endpoint.', 'feedvalue' ); ?>
		</p>
		<?php
	}

	/**
	 * Sanitize API URL input.
	 *
	 * @param string $url The URL input.
	 * @return string Sanitized URL or empty string.
	 */
	public function sanitize_api_url( $url ) {
		$url = trim( $url );

		if ( empty( $url ) ) {
			return '';
		}

		// Validate URL format
		$url = esc_url_raw( $url );

		// Remove trailing slash
		$url = rtrim( $url, '/' );

		// Validate it's a proper URL
		if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
			add_settings_error(
				'feedvalue_api_url_override',
				'invalid_url',
				__( 'Invalid API URL format. Please enter a valid URL.', 'feedvalue' ),
				'error'
			);
			return '';
		}

		return $url;
	}
}
