<?php
/**
 * AJAX handlers for admin actions.
 *
 * @package FeedValue
 */

class FeedValue_Ajax {

	/**
	 * Test API connection.
	 */
	public function test_connection() {
		check_ajax_referer( 'feedvalue_test_connection', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$api_key_input = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';

		// Check if the input is the placeholder (masked key) - use stored key instead
		if ( empty( $api_key_input ) || '••••••••••••••••' === $api_key_input ) {
			$settings      = new FeedValue_Settings();
			$encrypted_key = get_option( 'feedvalue_api_key', '' );

			if ( empty( $encrypted_key ) ) {
				wp_send_json_error( array( 'message' => __( 'API key is required. Please enter your API key first.', 'feedvalue' ) ) );
			}

			$api_key = $settings->decrypt_api_key( $encrypted_key );
		} else {
			$api_key = $api_key_input;
		}

		if ( empty( $api_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key is required.', 'feedvalue' ) ) );
		}

		$api_client = new FeedValue_API_Client();
		$result     = $api_client->test_connection( $api_key );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( array( 'message' => __( 'Connection successful!', 'feedvalue' ) ) );
	}

	/**
	 * Refresh widgets from API.
	 */
	public function refresh_widgets() {
		check_ajax_referer( 'feedvalue_refresh_widgets', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		// Get API key
		$settings   = new FeedValue_Settings();
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		if ( empty( $encrypted_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key not configured.', 'feedvalue' ) ) );
		}

		$api_key = $settings->decrypt_api_key( $encrypted_key );

		// Clear cache
		$cache = new FeedValue_Cache();
		$cache->delete( 'widgets_cache' );

		// Fetch fresh widgets
		$api_client = new FeedValue_API_Client();
		$widgets    = $api_client->get_widgets( $api_key );

		if ( is_wp_error( $widgets ) ) {
			wp_send_json_error( array( 'message' => $widgets->get_error_message() ) );
		}

		// Cache the widgets
		$cache->set( 'widgets_cache', $widgets );

		wp_send_json_success(
			array(
				'message' => __( 'Widgets refreshed successfully!', 'feedvalue' ),
				'count'   => count( $widgets ),
			)
		);
	}

	/**
	 * Clear all caches.
	 */
	public function clear_cache() {
		check_ajax_referer( 'feedvalue_clear_cache', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$cache = new FeedValue_Cache();
		$cache->clear_all();

		wp_send_json_success( array( 'message' => __( 'Cache cleared successfully!', 'feedvalue' ) ) );
	}

	/**
	 * Get targeting rules for a widget.
	 */
	public function get_targeting_rules() {
		check_ajax_referer( 'feedvalue_save_targeting', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$widget_id = isset( $_POST['widget_id'] ) ? sanitize_text_field( wp_unslash( $_POST['widget_id'] ) ) : '';

		if ( empty( $widget_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Widget ID is required.', 'feedvalue' ) ) );
		}

		$widget_manager = new FeedValue_Widget_Manager();
		$rules          = $widget_manager->get_targeting_rules( $widget_id );

		wp_send_json_success( array( 'rules' => $rules ) );
	}

	/**
	 * Save targeting rules for a widget.
	 */
	public function save_targeting_rules() {
		check_ajax_referer( 'feedvalue_save_targeting', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$widget_id = isset( $_POST['widget_id'] ) ? sanitize_text_field( wp_unslash( $_POST['widget_id'] ) ) : '';
		$rules     = isset( $_POST['rules'] ) ? json_decode( wp_unslash( $_POST['rules'] ), true ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $widget_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Widget ID is required.', 'feedvalue' ) ) );
		}

		// Sanitize rules
		$sanitized_rules = $this->sanitize_targeting_rules( $rules );

		// Save rules
		$widget_manager = new FeedValue_Widget_Manager();
		$result         = $widget_manager->save_targeting_rules( $widget_id, $sanitized_rules );

		if ( false === $result ) {
			// update_option returns false if value unchanged - check if data matches
			$current_rules = $widget_manager->get_targeting_rules( $widget_id );
			if ( $current_rules === $sanitized_rules ) {
				// Data is the same, not an error
				wp_send_json_success( array( 'message' => __( 'Targeting rules saved.', 'feedvalue' ) ) );
			}
			wp_send_json_error( array( 'message' => __( 'Failed to save targeting rules.', 'feedvalue' ) ) );
		}

		wp_send_json_success( array( 'message' => __( 'Targeting rules saved successfully!', 'feedvalue' ) ) );
	}

	/**
	 * Sanitize targeting rules.
	 *
	 * @param array $rules Targeting rules.
	 * @return array Sanitized rules.
	 */
	private function sanitize_targeting_rules( $rules ) {
		if ( ! is_array( $rules ) ) {
			return array();
		}

		$sanitized = array();

		foreach ( array( 'inclusion', 'exclusion' ) as $rule_type ) {
			if ( ! isset( $rules[ $rule_type ] ) || ! is_array( $rules[ $rule_type ] ) ) {
				continue;
			}

			$sanitized[ $rule_type ] = array();

			foreach ( $rules[ $rule_type ] as $rule ) {
				if ( ! is_array( $rule ) || ! isset( $rule['type'] ) ) {
					continue;
				}

				$sanitized_rule = array(
					'type' => sanitize_text_field( $rule['type'] ),
				);

				if ( isset( $rule['value'] ) ) {
					if ( is_array( $rule['value'] ) ) {
						$sanitized_rule['value'] = array_map( 'sanitize_text_field', $rule['value'] );
					} else {
						$sanitized_rule['value'] = sanitize_text_field( $rule['value'] );
					}
				}

				$sanitized[ $rule_type ][] = $sanitized_rule;
			}
		}

		return $sanitized;
	}

	/**
	 * Get feedback list.
	 */
	public function get_feedback() {
		check_ajax_referer( 'feedvalue_feedback', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		// Get API key
		$settings      = new FeedValue_Settings();
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		if ( empty( $encrypted_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key not configured. Please configure your API key in Settings.', 'feedvalue' ) ) );
		}

		$api_key = $settings->decrypt_api_key( $encrypted_key );

		// Build filters from request
		$filters = array();
		if ( ! empty( $_POST['status'] ) ) {
			$filters['status'] = sanitize_text_field( wp_unslash( $_POST['status'] ) );
		}
		if ( ! empty( $_POST['limit'] ) ) {
			$filters['limit'] = absint( $_POST['limit'] );
		}
		if ( ! empty( $_POST['offset'] ) ) {
			$filters['offset'] = absint( $_POST['offset'] );
		}

		$api_client = new FeedValue_API_Client();
		$feedback   = $api_client->get_feedback( $api_key, $filters );

		if ( is_wp_error( $feedback ) ) {
			wp_send_json_error( array( 'message' => $feedback->get_error_message() ) );
		}

		wp_send_json_success( array( 'feedback' => $feedback ) );
	}

	/**
	 * Get single feedback details.
	 */
	public function get_feedback_single() {
		check_ajax_referer( 'feedvalue_feedback', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$submission_id = isset( $_POST['submission_id'] ) ? sanitize_text_field( wp_unslash( $_POST['submission_id'] ) ) : '';

		if ( empty( $submission_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Submission ID is required.', 'feedvalue' ) ) );
		}

		// Get API key
		$settings      = new FeedValue_Settings();
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		if ( empty( $encrypted_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key not configured.', 'feedvalue' ) ) );
		}

		$api_key = $settings->decrypt_api_key( $encrypted_key );

		$api_client = new FeedValue_API_Client();
		$feedback   = $api_client->get_single_feedback( $api_key, $submission_id );

		if ( is_wp_error( $feedback ) ) {
			wp_send_json_error( array( 'message' => $feedback->get_error_message() ) );
		}

		wp_send_json_success( array( 'feedback' => $feedback ) );
	}

	/**
	 * Update feedback status.
	 */
	public function update_feedback_status() {
		check_ajax_referer( 'feedvalue_feedback', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$submission_id = isset( $_POST['submission_id'] ) ? sanitize_text_field( wp_unslash( $_POST['submission_id'] ) ) : '';
		$status        = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : '';

		if ( empty( $submission_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Submission ID is required.', 'feedvalue' ) ) );
		}

		if ( empty( $status ) ) {
			wp_send_json_error( array( 'message' => __( 'Status is required.', 'feedvalue' ) ) );
		}

		// Get API key
		$settings      = new FeedValue_Settings();
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		if ( empty( $encrypted_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key not configured.', 'feedvalue' ) ) );
		}

		$api_key = $settings->decrypt_api_key( $encrypted_key );

		$api_client = new FeedValue_API_Client();
		$feedback   = $api_client->update_feedback_status( $api_key, $submission_id, $status );

		if ( is_wp_error( $feedback ) ) {
			wp_send_json_error( array( 'message' => $feedback->get_error_message() ) );
		}

		wp_send_json_success(
			array(
				'message'  => __( 'Status updated successfully.', 'feedvalue' ),
				'feedback' => $feedback,
			)
		);
	}

	/**
	 * Delete feedback.
	 */
	public function delete_feedback() {
		check_ajax_referer( 'feedvalue_feedback', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$submission_id = isset( $_POST['submission_id'] ) ? sanitize_text_field( wp_unslash( $_POST['submission_id'] ) ) : '';

		if ( empty( $submission_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Submission ID is required.', 'feedvalue' ) ) );
		}

		// Get API key
		$settings      = new FeedValue_Settings();
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		if ( empty( $encrypted_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key not configured.', 'feedvalue' ) ) );
		}

		$api_key = $settings->decrypt_api_key( $encrypted_key );

		$api_client = new FeedValue_API_Client();
		$result     = $api_client->delete_feedback( $api_key, $submission_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( array( 'message' => __( 'Feedback deleted successfully.', 'feedvalue' ) ) );
	}

	/**
	 * Bulk feedback action.
	 */
	public function bulk_feedback_action() {
		check_ajax_referer( 'feedvalue_feedback', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$action         = isset( $_POST['bulk_action'] ) ? sanitize_text_field( wp_unslash( $_POST['bulk_action'] ) ) : '';
		$submission_ids = isset( $_POST['submission_ids'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['submission_ids'] ) ) : array();

		if ( empty( $action ) ) {
			wp_send_json_error( array( 'message' => __( 'Action is required.', 'feedvalue' ) ) );
		}

		if ( empty( $submission_ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No items selected.', 'feedvalue' ) ) );
		}

		// Get API key
		$settings      = new FeedValue_Settings();
		$encrypted_key = get_option( 'feedvalue_api_key', '' );

		if ( empty( $encrypted_key ) ) {
			wp_send_json_error( array( 'message' => __( 'API key not configured.', 'feedvalue' ) ) );
		}

		$api_key    = $settings->decrypt_api_key( $encrypted_key );
		$api_client = new FeedValue_API_Client();

		$success_count = 0;
		$error_count   = 0;

		foreach ( $submission_ids as $submission_id ) {
			if ( 'delete' === $action ) {
				$result = $api_client->delete_feedback( $api_key, $submission_id );
			} else {
				// Map bulk action to status
				$status_map = array(
					'read'     => 'read',
					'resolved' => 'resolved',
					'archived' => 'archived',
				);

				if ( ! isset( $status_map[ $action ] ) ) {
					continue;
				}

				$result = $api_client->update_feedback_status( $api_key, $submission_id, $status_map[ $action ] );
			}

			if ( is_wp_error( $result ) ) {
				++$error_count;
			} else {
				++$success_count;
			}
		}

		if ( $error_count > 0 && $success_count === 0 ) {
			wp_send_json_error( array( 'message' => __( 'All operations failed.', 'feedvalue' ) ) );
		}

		wp_send_json_success(
			array(
				'message'       => sprintf(
					/* translators: %d: number of items processed */
					_n( '%d item processed successfully.', '%d items processed successfully.', $success_count, 'feedvalue' ),
					$success_count
				),
				'success_count' => $success_count,
				'error_count'   => $error_count,
			)
		);
	}

	/**
	 * Save widget settings (enabled state and global position).
	 */
	public function save_widget_settings() {
		check_ajax_referer( 'feedvalue_save_widget_settings', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'feedvalue' ) ) );
		}

		$widget_id       = isset( $_POST['widget_id'] ) ? sanitize_text_field( wp_unslash( $_POST['widget_id'] ) ) : '';
		$enabled         = isset( $_POST['enabled'] ) ? ( $_POST['enabled'] === 'true' || $_POST['enabled'] === '1' ) : false;
		$global_position = isset( $_POST['global_position'] ) ? sanitize_text_field( wp_unslash( $_POST['global_position'] ) ) : '';

		if ( empty( $widget_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Widget ID is required.', 'feedvalue' ) ) );
		}

		// Validate global_position
		$allowed_positions = array( '', 'header', 'footer' );
		if ( ! in_array( $global_position, $allowed_positions, true ) ) {
			$global_position = '';
		}

		// Get existing settings
		$settings = get_option( 'feedvalue_widget_settings', array() );

		// Update this widget's settings
		if ( ! isset( $settings[ $widget_id ] ) ) {
			$settings[ $widget_id ] = array();
		}

		$settings[ $widget_id ]['enabled']         = $enabled;
		$settings[ $widget_id ]['global_position'] = $global_position;

		// Save settings
		$result = update_option( 'feedvalue_widget_settings', $settings );

		if ( false === $result ) {
			// Check if the value is the same (update_option returns false if unchanged)
			$current = get_option( 'feedvalue_widget_settings', array() );
			if ( $current === $settings ) {
				wp_send_json_success( array( 'message' => __( 'Settings saved.', 'feedvalue' ) ) );
			}
			wp_send_json_error( array( 'message' => __( 'Failed to save settings.', 'feedvalue' ) ) );
		}

		wp_send_json_success( array( 'message' => __( 'Settings saved.', 'feedvalue' ) ) );
	}
}
