<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package FeedValue
 */

class FeedValue_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @var string
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Register the stylesheets for the admin area.
	 */
	public function enqueue_styles() {
		$screen = get_current_screen();

		// Only load on FeedValue admin pages
		if ( strpos( $screen->id, 'feedvalue' ) === false ) {
			return;
		}

		// Load Google Fonts (DM Sans + JetBrains Mono)
		wp_enqueue_style(
			$this->plugin_name . '-fonts',
			'https://fonts.googleapis.com/css2?family=DM+Sans:ital,opsz,wght@0,9..40,100..1000;1,9..40,100..1000&family=JetBrains+Mono:wght@400;500;600&display=swap',
			array(),
			null,
			'all'
		);

		wp_enqueue_style(
			$this->plugin_name,
			FEEDVALUE_PLUGIN_URL . 'admin/css/feedvalue-admin.css',
			array( $this->plugin_name . '-fonts' ),
			$this->version,
			'all'
		);
	}

	/**
	 * Register the JavaScript for the admin area.
	 */
	public function enqueue_scripts() {
		$screen = get_current_screen();

		// Only load on FeedValue admin pages
		if ( strpos( $screen->id, 'feedvalue' ) === false ) {
			return;
		}

		wp_enqueue_script(
			$this->plugin_name,
			FEEDVALUE_PLUGIN_URL . 'admin/js/feedvalue-admin.js',
			array( 'jquery' ),
			$this->version,
			true  // Load in footer to ensure DOM is ready
		);

		// Localize script with AJAX URL and nonces
		wp_localize_script(
			$this->plugin_name,
			'feedvalueAdmin',
			array(
				'ajaxurl'                 => admin_url( 'admin-ajax.php' ),
				'testConnectionNonce'     => wp_create_nonce( 'feedvalue_test_connection' ),
				'refreshWidgetsNonce'     => wp_create_nonce( 'feedvalue_refresh_widgets' ),
				'clearCacheNonce'         => wp_create_nonce( 'feedvalue_clear_cache' ),
				'saveTargetingNonce'      => wp_create_nonce( 'feedvalue_save_targeting' ),
				'saveWidgetSettingsNonce' => wp_create_nonce( 'feedvalue_save_widget_settings' ),
				'strings'                 => array(
					'testing'           => __( 'Testing connection...', 'feedvalue' ),
					'refreshing'        => __( 'Refreshing widgets...', 'feedvalue' ),
					'clearing'          => __( 'Clearing cache...', 'feedvalue' ),
					'saving'            => __( 'Saving...', 'feedvalue' ),
					'saved'             => __( 'Saved!', 'feedvalue' ),
					'error'             => __( 'An error occurred. Please try again.', 'feedvalue' ),
					'confirmClearCache' => __( 'Are you sure you want to clear all cached data?', 'feedvalue' ),
				),
			)
		);

		// Load feedback page JS on main FeedValue page
		if ( 'toplevel_page_feedvalue' === $screen->id ) {
			wp_enqueue_script(
				$this->plugin_name . '-feedback',
				FEEDVALUE_PLUGIN_URL . 'admin/js/feedvalue-feedback.js',
				array( 'jquery' ),
				$this->version,
				true
			);

			// Check if API key is configured
			$encrypted_key = get_option( 'feedvalue_api_key', '' );
			$is_configured = ! empty( $encrypted_key );

			wp_localize_script(
				$this->plugin_name . '-feedback',
				'feedvalueFeedback',
				array(
					'ajaxurl'      => admin_url( 'admin-ajax.php' ),
					'nonce'        => wp_create_nonce( 'feedvalue_feedback' ),
					'isConfigured' => $is_configured,
					'strings' => array(
						'loading'           => __( 'Loading feedback...', 'feedvalue' ),
						'error'             => __( 'An error occurred. Please try again.', 'feedvalue' ),
						'noFeedback'        => __( 'No feedback yet', 'feedvalue' ),
						'noFeedbackDesc'    => __( 'Feedback will appear here once users start submitting through your widgets.', 'feedvalue' ),
						'noResults'         => __( 'No matching feedback', 'feedvalue' ),
						'noResultsDesc'     => __( 'Try adjusting your filters or search query.', 'feedvalue' ),
						'confirmDelete'     => __( 'Are you sure you want to delete this feedback?', 'feedvalue' ),
						'confirmBulkDelete' => __( 'Are you sure you want to delete the selected feedback items?', 'feedvalue' ),
						'selectAction'      => __( 'Please select an action.', 'feedvalue' ),
						'selectItems'       => __( 'Please select at least one item.', 'feedvalue' ),
						'processing'        => __( 'Processing...', 'feedvalue' ),
						'apply'             => __( 'Apply', 'feedvalue' ),
						'readMore'          => __( 'Read more', 'feedvalue' ),
						'showLess'          => __( 'Show less', 'feedvalue' ),
						'anonymous'         => __( 'Anonymous', 'feedvalue' ),
						'markReviewed'      => __( 'Mark Reviewed', 'feedvalue' ),
						'archive'           => __( 'Archive', 'feedvalue' ),
						'delete'            => __( 'Delete', 'feedvalue' ),
						'selectedCount'     => __( '%d selected', 'feedvalue' ),
						'showingItems'      => __( 'Showing %1$d-%2$d of %3$d items', 'feedvalue' ),
						'noDataExport'      => __( 'No data to export.', 'feedvalue' ),
						'statusNew'         => __( 'New', 'feedvalue' ),
						'statusRead'        => __( 'Read', 'feedvalue' ),
						'statusResolved'    => __( 'Resolved', 'feedvalue' ),
						'statusArchived'    => __( 'Archived', 'feedvalue' ),
						'demoUpdated'       => __( 'Demo updated (connect your API key to save changes)', 'feedvalue' ),
						'demoDeleted'       => __( 'Demo item removed (connect your API key to save changes)', 'feedvalue' ),
						'demoBulkUpdated'   => __( 'Demo items updated (connect your API key to save changes)', 'feedvalue' ),
					),
				)
			);
		}
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		$capability = apply_filters( 'feedvalue_admin_capability', 'manage_options' );

		// Main menu page - points to Feedback as default
		add_menu_page(
			__( 'FeedValue', 'feedvalue' ),
			__( 'FeedValue', 'feedvalue' ),
			$capability,
			'feedvalue',
			array( $this, 'display_feedback_page' ),
			'dashicons-feedback',
			80
		);

		// Feedback submenu (first item, same slug as parent)
		add_submenu_page(
			'feedvalue',
			__( 'Feedback', 'feedvalue' ),
			__( 'Feedback', 'feedvalue' ),
			$capability,
			'feedvalue',
			array( $this, 'display_feedback_page' )
		);

		// Settings submenu (second item)
		add_submenu_page(
			'feedvalue',
			__( 'FeedValue Settings', 'feedvalue' ),
			__( 'Settings', 'feedvalue' ),
			$capability,
			'feedvalue-settings',
			array( $this, 'display_admin_page' )
		);

		// Help submenu (third item)
		add_submenu_page(
			'feedvalue',
			__( 'FeedValue Help', 'feedvalue' ),
			__( 'Help', 'feedvalue' ),
			$capability,
			'feedvalue-help',
			array( $this, 'display_help_page' )
		);
	}

	/**
	 * Display the feedback page.
	 */
	public function display_feedback_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'feedvalue' ) );
		}

		require_once FEEDVALUE_PLUGIN_DIR . 'admin/partials/feedvalue-feedback-display.php';
	}

	/**
	 * Display the admin page.
	 */
	public function display_admin_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'feedvalue' ) );
		}

		require_once FEEDVALUE_PLUGIN_DIR . 'admin/partials/feedvalue-admin-display.php';
	}

	/**
	 * Display the help page.
	 */
	public function display_help_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'feedvalue' ) );
		}

		require_once FEEDVALUE_PLUGIN_DIR . 'admin/partials/feedvalue-help-display.php';
	}

	/**
	 * Add action links to plugin list page.
	 *
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_action_links( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			esc_url( admin_url( 'admin.php?page=feedvalue-settings' ) ),
			__( 'Settings', 'feedvalue' )
		);

		array_unshift( $links, $settings_link );

		return $links;
	}

	/**
	 * Add row meta links to plugin list page (shown after version number).
	 *
	 * @param array  $links Plugin row meta links.
	 * @param string $file  Plugin file path.
	 * @return array Modified row meta links.
	 */
	public function add_row_meta_links( $links, $file ) {
		if ( plugin_basename( FEEDVALUE_PLUGIN_FILE ) !== $file ) {
			return $links;
		}

		$row_meta = array(
			'docs'    => sprintf(
				'<a href="%s" target="_blank" rel="noopener noreferrer" aria-label="%s">%s</a>',
				esc_url( 'https://feedvalue.com/docs/wordpress-plugin' ),
				esc_attr__( 'View FeedValue documentation', 'feedvalue' ),
				esc_html__( 'Documentation', 'feedvalue' )
			),
			'support' => sprintf(
				'<a href="%s" target="_blank" rel="noopener noreferrer" aria-label="%s">%s</a>',
				esc_url( 'https://feedvalue.com/support' ),
				esc_attr__( 'Get FeedValue support', 'feedvalue' ),
				esc_html__( 'Support', 'feedvalue' )
			),
			'faq'     => sprintf(
				'<a href="%s" target="_blank" rel="noopener noreferrer" aria-label="%s">%s</a>',
				esc_url( 'https://feedvalue.com/docs/faq' ),
				esc_attr__( 'View frequently asked questions', 'feedvalue' ),
				esc_html__( 'FAQ', 'feedvalue' )
			),
		);

		return array_merge( $links, $row_meta );
	}
}
